/*  This routine "executes" the field selected by the HCI.  The input is the field_ID.
     The routine looks up the current HCI state and transfers control to the appropriate routine in the ADAU1466_cmd library.
     This routine is called whenever the value of a field is updated and it is called on power-up to restore the hardware state
*/

#include "ADAU1466_Lookup_tables.h"
#include "ADAU1466_cmd.h"
#include "Cell_map.h"


ADAU1466_cmd ADAU1466_cmd(ADAU1466_Mute_addresses);  //start an instance of the cmd library using the mute addresses


void update_from_HCI(byte field_ID, int value) {

  Serial.print("Decode command  ");
  Serial.print(field_ID);
  Serial.print("   Value:  ");
  Serial.println(value);

  //This is called from the HCI update routine and the restore routine.  Determine what to do for the updated field
  switch (field_ID) {

    case Main_volume:
      ADAU1466_cmd.Main_Volume(Multiple_1, value);
      break;

    case Mute:
      //ADAU1466_cmd.Mute_all(ADAU1466_Mute_addresses, value);
      break;

    case Input:
      /*The Act 2 board has a multiplexer controlled by pin 5 of the micro, but the Input_select library routine only
      controls the multiplexer inside the ADAU1466.  So, we need to check for the values that control the external
      mux and make that selection here.  Then, we can call the routine to control the ADAU1466 mux.
      */
      switch (value) {
        case 0:  //Analog
          ADAU1466_cmd.Input_select(Input_SW, 0);
          break;
        case 1:                           //SPDIF 1 (WiFi streamer)
          digitalWrite(SPDIF_sel, HIGH);  //External mux = HIGH for WiFi input
          //Serial.println("SPDIF 1 selected");
          ADAU1466_cmd.Input_select(Input_SW, 1);
          break;
        case 2:                          //SPDIF 2 (PC from HDMI extractor)
          digitalWrite(SPDIF_sel, LOW);  //External mux = HIGH for WiFi input
          ADAU1466_cmd.Input_select(Input_SW, 1);
          //Serial.println("SPDIF 2 selected");
          break;
        case 3:  //Test tones input
          ADAU1466_cmd.Input_select(Input_SW, 2);
          break;
      }
      break;

    case chan_vol_T:
      ADAU1466_cmd.Chan_vol_trim(Tweeter_Trim, value);  //get trim level and format it for ADAU1466
      break;

    case chan_vol_W:
      ADAU1466_cmd.Chan_vol_trim(Woofer_Trim, value);
      break;

    case chan_vol_S:
      ADAU1466_cmd.Chan_vol_trim(Sub_Trim, value);
      break;

    case Delay_T:
      ADAU1466_cmd.Update_Delay(T_Delay, value);
      break;

    case Delay_W:
      ADAU1466_cmd.Update_Delay(W_Delay, value);
      break;

    case Delay_S:
      ADAU1466_cmd.Update_Delay(S_Delay, value);
      break;

    case Xover_T_W_type:
      ADAU1466_cmd.T_W_Xover(HCI_state[Xover_T_W_freq].value, HCI_state[Xover_T_W_type].value, biquad_T_W_addresses);  //use the HCI values and the address list to update the crossover
      break;

    case Xover_T_W_freq:
      ADAU1466_cmd.T_W_Xover(HCI_state[Xover_T_W_freq].value, HCI_state[Xover_T_W_type].value, biquad_T_W_addresses);  //use the HCI values and the address list to update the crossover
      break;

    case Xover_W_S_type:
      ADAU1466_cmd.W_S_Xover(HCI_state[Xover_W_S_freq].value, HCI_state[Xover_W_S_type].value, biquad_W_S_addresses);  //use the HCI values and the address list to update the crossover
      break;

    case Xover_W_S_freq:
      ADAU1466_cmd.W_S_Xover(HCI_state[Xover_W_S_freq].value, HCI_state[Xover_W_S_type].value, biquad_W_S_addresses);  //use the HCI values and the address list to update the crossover
      break;


    case Xover_T_W_polarity:
      ADAU1466_cmd.Xover_T_polarity(T_Polarity, HCI_state[Xover_T_W_polarity].value);
      break;

    case Xover_W_S_polarity:
      //ADAU1466_cmd.Xover_S_polarity(S_Polarity, HCI_state[Xover_W_S_polarity].value);
      break;

    case BSC_freq:
      ADAU1466_cmd.BSC_Freq_or_Gain(BSC, HCI_state[BSC_freq].value, HCI_state[BSC_Gain].value);
      break;

    case BSC_Gain:
      ADAU1466_cmd.BSC_Freq_or_Gain(BSC, HCI_state[BSC_freq].value, HCI_state[BSC_Gain].value);
      break;


    case EQ_32_Gain:
      ADAU1466_cmd.EQ_Filter(EQ_32, 32.0, HCI_state[field_ID].value);
      break;

    case EQ_64_Gain:
      ADAU1466_cmd.EQ_Filter(EQ_64, 64.0, HCI_state[field_ID].value);
      break;

    case EQ_125_Gain:
      ADAU1466_cmd.EQ_Filter(EQ_125, 125.0, HCI_state[field_ID].value);
      break;

    case EQ_250_Gain:
      ADAU1466_cmd.EQ_Filter(EQ_250, 250.0, HCI_state[field_ID].value);
      break;

    case EQ_500_Gain:
      ADAU1466_cmd.EQ_Filter(EQ_500, 500.0, HCI_state[field_ID].value);
      break;

    case EQ_1K_Gain:
      ADAU1466_cmd.EQ_Filter(EQ_1K, 1000.0, HCI_state[field_ID].value);
      break;

    case EQ_2K_Gain:
      ADAU1466_cmd.EQ_Filter(EQ_2K, 2000.0, HCI_state[field_ID].value);
      break;

    case EQ_4K_Gain:
      ADAU1466_cmd.EQ_Filter(EQ_4K, 4000.0, HCI_state[field_ID].value);
      break;

    case EQ_8K_Gain:
      ADAU1466_cmd.EQ_Filter(EQ_8K, 8000.0, HCI_state[field_ID].value);
      break;

    case EQ_16K_Gain:
      ADAU1466_cmd.EQ_Filter(EQ_16K, 16000.0, HCI_state[field_ID].value);
      break;

    case EQ_reset_Gain:
      /*  This command is implemented in the host, by repeatedly calling the EQ values with default values
        This command is only needed if a local display is used
        break;
     */
      break;

    case Rumble_filter_freq:
      ADAU1466_cmd.Rumble_filter(Rumble_biquad, HCI_state[Rumble_filter_freq].value, HCI_state[Rumble_filter_Q].value, BE.Rumble_filter_enabled);
      break;

    case Rumble_filter_Q:
      ADAU1466_cmd.Rumble_filter(Rumble_biquad, HCI_state[Rumble_filter_freq].value, HCI_state[Rumble_filter_Q].value, BE.Rumble_filter_enabled);
      break;

    case Peak_filter_freq:
      ADAU1466_cmd.Peak_filter(Peaking_biquad, HCI_state[Peak_filter_freq].value, HCI_state[Peak_filter_Gain].value, BE.Peak_filter_enabled);
      break;

    case Peak_filter_Gain:
      ADAU1466_cmd.Peak_filter(Peaking_biquad, HCI_state[Peak_filter_Gain].value, HCI_state[Peak_filter_Gain].value, BE.Peak_filter_enabled);
      break;

    case Curvature:
      ADAU1466_cmd.Curvature_select(HCI_state[Curvature].value, tweeter_delay_addresses, woofer_delay_addresses);
      break;

    case Shading:
      ADAU1466_cmd.Shading_select(HCI_state[Shading].value, tweeter_shading_addresses, woofer_shading_addresses);
      break;


    case Custom_filter_select:
      //this commmand is only used for integrated displays than need a "selector switch" to pick the custom filter
      //switch (HCI_state[field_ID].value) {}
      break;

      /*  Sigh...
         The ADAU1466 command library has two types of interfaces rather than one.  Most of the commands are of the form:  "command(index)".  The index is used to look up the actual
         parameters from the lists in Lookup_tables.h.  This form works well for some of the commands where the associated parameters need to be generated using Excel or other tools.
         The second form is "command(parameters)", where the actual parameters are converted from the text in the HCI display and passed as floating point values.  Both forms are
         useful, and both have maitenance issues.  Passing the index means that there must be a list that converts the index to the parameters, and if the HCI changes, this list must
         be updated as well.  That means the Command library needs to "know" too much about the HCI.  The other form, which passes the actual floating point parameters, is more logical
         for simple functions such as filters, where it is relatively easy to convert the value displayed by the HCI into floating point numbers.  However, this approach doesn't work
         well for conversions into HEX values or logarthically spaced parameters such as volume level, and it requires the HCI to use text values that can be easily converted to numbers.

         The Custom Filter commands in this Custom Filters section use the form of command(parameters).  Notice the "get_float_value" function in the parameter list--this function extracts
         the HCI text from HCI.h and converts it to floating point. 
    */

    case Custom_Shared1_Type:
      ADAU1466_cmd.Custom_filter(Custom_1, get_f_type(Custom_Shared1_Type), get_float_value(Custom_Shared1_Freq), get_float_value(Custom_Shared1_Gain), get_float_value(Custom_Shared1_Q));
      break;

    case Custom_Shared1_Freq:
      ADAU1466_cmd.Custom_filter(Custom_1, get_f_type(Custom_Shared1_Type), get_float_value(Custom_Shared1_Freq), get_float_value(Custom_Shared1_Gain), get_float_value(Custom_Shared1_Q));
      break;

    case Custom_Shared1_Q:
      ADAU1466_cmd.Custom_filter(Custom_1, get_f_type(Custom_Shared1_Type), get_float_value(Custom_Shared1_Freq), get_float_value(Custom_Shared1_Gain), get_float_value(Custom_Shared1_Q));
      break;

    case Custom_Shared1_Gain:
      ADAU1466_cmd.Custom_filter(Custom_1, get_f_type(Custom_Shared1_Type), get_float_value(Custom_Shared1_Freq), get_float_value(Custom_Shared1_Gain), get_float_value(Custom_Shared1_Q));
      break;

    case Custom_Tweeter_Type:
      ADAU1466_cmd.Custom_filter(Cust_Tweeter, get_f_type(Custom_Tweeter_Type), get_float_value(Custom_Tweeter_Freq), get_float_value(Custom_Tweeter_Gain), get_float_value(Custom_Tweeter_Q));
      break;

    case Custom_Tweeter_Freq:
      ADAU1466_cmd.Custom_filter(Cust_Tweeter, get_f_type(Custom_Tweeter_Type), get_float_value(Custom_Tweeter_Freq), get_float_value(Custom_Tweeter_Gain), get_float_value(Custom_Tweeter_Q));
      break;

    case Custom_Tweeter_Q:
      ADAU1466_cmd.Custom_filter(Cust_Tweeter, get_f_type(Custom_Tweeter_Type), get_float_value(Custom_Tweeter_Freq), get_float_value(Custom_Tweeter_Gain), get_float_value(Custom_Tweeter_Q));
      break;

    case Custom_Tweeter_Gain:
      ADAU1466_cmd.Custom_filter(Cust_Tweeter, get_f_type(Custom_Tweeter_Type), get_float_value(Custom_Tweeter_Freq), get_float_value(Custom_Tweeter_Gain), get_float_value(Custom_Tweeter_Q));
      break;


    case Custom_Woofer_Type:
      ADAU1466_cmd.Custom_filter(Cust_Woofer, get_f_type(Custom_Woofer_Type), get_float_value(Custom_Woofer_Freq), get_float_value(Custom_Woofer_Gain), get_float_value(Custom_Woofer_Q));
      break;

    case Custom_Woofer_Freq:
      ADAU1466_cmd.Custom_filter(Cust_Woofer, get_f_type(Custom_Woofer_Type), get_float_value(Custom_Woofer_Freq), get_float_value(Custom_Woofer_Gain), get_float_value(Custom_Woofer_Q));
      break;

    case Custom_Woofer_Q:
      ADAU1466_cmd.Custom_filter(Cust_Woofer, get_f_type(Custom_Woofer_Type), get_float_value(Custom_Woofer_Freq), get_float_value(Custom_Woofer_Gain), get_float_value(Custom_Woofer_Q));
      break;

    case Custom_Woofer_Gain:
      ADAU1466_cmd.Custom_filter(Cust_Woofer, get_f_type(Custom_Woofer_Type), get_float_value(Custom_Woofer_Freq), get_float_value(Custom_Woofer_Gain), get_float_value(Custom_Woofer_Q));
      break;


    case Custom_Sub_Type:
      ADAU1466_cmd.Custom_filter(Custom_Sub, get_f_type(Custom_Sub_Type), get_float_value(Custom_Sub_Freq), get_float_value(Custom_Sub_Gain), get_float_value(Custom_Sub_Q));
      break;

    case Custom_Sub_Freq:
      ADAU1466_cmd.Custom_filter(Custom_Sub, get_f_type(Custom_Sub_Type), get_float_value(Custom_Sub_Freq), get_float_value(Custom_Sub_Gain), get_float_value(Custom_Sub_Q));
      break;

    case Custom_Sub_Q:
      ADAU1466_cmd.Custom_filter(Custom_Sub, get_f_type(Custom_Sub_Type), get_float_value(Custom_Sub_Freq), get_float_value(Custom_Sub_Gain), get_float_value(Custom_Sub_Q));
      break;

    case Custom_Sub_Gain:
      ADAU1466_cmd.Custom_filter(Custom_Sub, get_f_type(Custom_Sub_Type), get_float_value(Custom_Sub_Freq), get_float_value(Custom_Sub_Gain), get_float_value(Custom_Sub_Q));
      break;


    case Bass_Enhancement_Mode:

      switch (HCI_state[field_ID].value) {

        case 0:  //this is Rumble On
          BE.Rumble_filter_enabled = true;
          BE.Peak_filter_enabled = false;
          BE.SuperBass_enabled = false;
          //Now update the Bass Enhancement filter status
          BE_update();
          break;

        case 1:  //this is Peak On
          BE.Rumble_filter_enabled = false;
          BE.Peak_filter_enabled = true;
          BE.SuperBass_enabled = false;
          BE_update();
          break;

        case 2:  //this is Rumble and Peak On
          BE.Rumble_filter_enabled = true;
          BE.Peak_filter_enabled = true;
          BE.SuperBass_enabled = false;
          BE_update();
          break;

        case 3:  //this is SuperBass On
          BE.Rumble_filter_enabled = false;
          BE.Peak_filter_enabled = false;
          BE.SuperBass_enabled = true;
          BE_update();
          break;

        case 4:  //this is Off
          BE.Rumble_filter_enabled = false;
          BE.Peak_filter_enabled = false;
          BE.SuperBass_enabled = false;
          BE_update();
          break;
      }


    case ADAU1466_Source:
      ADAU1466_cmd.ADAU1466_Source_select(Source_SW, HCI_state[ADAU1466_Source].value, Sweep1);
      break;

    case ADAU1466_Freq:
      ADAU1466_cmd.ADAU1466_Sine(Tone1, HCI_state[ADAU1466_Freq].value);
      break;

    case ADAU1466_Range:
      ADAU1466_cmd.ADAU1466_Sweep(Sweep1, HCI_state[ADAU1466_Range].value, HCI_state[ADAU1466_Steps].value, HCI_state[ADAU1466_Speed].value);
      break;

    case ADAU1466_Steps:
      ADAU1466_cmd.ADAU1466_Sweep(Sweep1, HCI_state[ADAU1466_Range].value, HCI_state[ADAU1466_Steps].value, HCI_state[ADAU1466_Speed].value);
      break;

    case ADAU1466_Speed:
      ADAU1466_cmd.ADAU1466_Sweep(Sweep1, HCI_state[ADAU1466_Range].value, HCI_state[ADAU1466_Steps].value, HCI_state[ADAU1466_Speed].value);
      break;

    case Speaker_select:
      ADAU1466_cmd.Speaker_select(HCI_state[Speaker_select].value, ADAU1466_Mute_addresses, tweeter_mute_addresses, woofer_mute_addresses);
      break;

    default:
      break;
  }
}

/* Support Routines for the HW_Updates

*/


/*________________________________________________________________________________________________________

  ADAU1466 Mute
  ________________________________________________________________________________________________________
*/

void ADAU1466_mute(bool _muted) {
  if (_muted == true) {
    initialization_mute();
    Serial.println("muted");
  } else {
    initialization_unmute();
    Serial.println("unmuted");
  }
}

void initialization_mute() {
  ADAU1466_cmd.Mute_all(ADAU1466_Mute_addresses, 1);
}

void initialization_unmute() {
  ADAU1466_cmd.Mute_all(ADAU1466_Mute_addresses, 0);
}



/*  This routine uses the field ID to look up the caption string for the HCI display.
     It then converts the caption string to a floating point value.  This routine
     works for Freq, Q, and Gain, where the caption is the actual value.
     This routine does not assume that the current_page and current_page_field variables
     are pointing to the right HCI field -- it uses just the field_ID
*/
float get_float_value(byte field_ID) {
  byte len;
  int k;
  char local_field_string[18];
  long table_offset;

  len = HCI_state[field_ID].value_length;
  table_offset = HCI_state[field_ID].value * (len + 1);

  for (k = 0; k < len; k++) {
    local_field_string[k] = pgm_read_byte_far(HCI_state[field_ID].value_table_pointer + k + table_offset);
  }
  local_field_string[k] = '\0';
  return atof(local_field_string);
}

/* call the following routine by using the "Type" for that filter
    example:  get_filter_type(Custom_Shared1_Type) for the following routines:  Custom_Shared1_freq, Custom_Shared1_Q, Custom_Shared1_Gain, etc.
*/


Shared_enum::tfilters get_f_type(byte field_ID) {

  switch (HCI_state[field_ID].value) {
    case 0: return Shared_enum::One_Pole_LP;
    case 1: return Shared_enum::One_Pole_HP;
    case 2: return Shared_enum::Two_Pole_LP;
    case 3: return Shared_enum::Two_Pole_HP;
    case 4: return Shared_enum::LP_Shelf;
    case 5: return Shared_enum::HP_Shelf;
    case 6: return Shared_enum::Bandpass;
    case 7: return Shared_enum::Notch;
    case 8: return Shared_enum::Peaking;
    case 9: return Shared_enum::All_Pass;
    case 10: return Shared_enum::None;
    default: return Shared_enum::None;
  }
}

void BE_update() {
  /*
  ADAU1466_cmd.Rumble_filter(Rumble_biquad, HCI_state[Rumble_filter_freq].value, HCI_state[Rumble_filter_Q].value, BE.Rumble_filter_enabled);
  ADAU1466_cmd.Peak_filter(Peaking_biquad, HCI_state[Peak_filter_freq].value, HCI_state[Peak_filter_Gain].value, BE.Peak_filter_enabled);
  ADAU1466_cmd.SuperBass_filter(SuperBass, HCI_state[SuperBass_freq].value, HCI_state[SuperBass_Intensity].value, HCI_state[SuperBass_Gain].value, BE.SuperBass_enabled);
  */
}