/*These routines check for an incoming character on the serial port, and then waits long enough for the entire
   message to be completed.  It then decodes the message by isolating the command, Parameter #1 and Parameter #2 (if used).
  The data is sent as ASCII characters.  Binary data is sent as two ASCII characters (MSB:LSB)
  that corresponds to the Hex value.  Example:  0x45 is sent as ASCII '4' (Dec 52) and '5' (Dec 53).

  The supported commands are:
    ?:  dead or alive test--no parameters
    Set I2C Address: A, MSB:LSB
    Set I2C MSB subaddress for 16-bit addresses: T, MSB:LSB
    Variable length I2C message: M, Count, MSB:LSB, .....
    Read from I2C address:  R, MSB:LSB
    Read a block from I2C addresses:  L, Count, MSB:LSB

    Note:  since the ADAU1466 uses 16-bit I2C subaddress,
    the "T" command must be used prior to the M or R commands

    Volume: Z, volume_code
    Channel Volume: C, Chan#, chan_vol_code
    Delay: X, Chan#, delay_code
    BSC: B, Freq_code, Gain_code
    Filter Spec, Woofer-Tweeter: F, Xover_type code, Xover_freq code
    EQ: E, EQ_freq_code, EQ_gain_code
    Input: I, Inputcode
    Parameter RAM Save/Load: P, Blockcode
    Mute: V, Mutecode
    Save/Restore:  S, Save current HCI settings in EEPROM
    Rumble:  Q, Rumble_code, Value
    SuperBass:  K, Superbass_field_code, value
    Dynamic Bass:  D, Dynamicbass_field_code, value
    Dipole_compensation:  G, Dipole_field_code, value
*/
#include <ADAU1466_SPI.h>  //Needed to echo commands to the SPDIF transmitter
extern ADAU1466_SPI Lib_SPI;

//this is how the commands are validated.  To add a new command, specify the code and make sure the number of commands is updated in the main.ino declarations
void CMD_init() {
  byte i;
  i = 0;
  CMD[i].command_code = 'S';
  CMD[i].command_length = 2;
  CMD[i].supported = true;
  i = 1;
  CMD[i].command_code = 'B';
  CMD[i].command_length = 3;
  CMD[i].supported = true;
  i = 2;
  CMD[i].command_code = 'Z';
  CMD[i].command_length = 2;
  CMD[i].supported = true;
  i = 3;
  CMD[i].command_code = 'F';
  CMD[i].command_length = 3;
  CMD[i].supported = true;
  i = 4;
  CMD[i].command_code = 'E';
  CMD[i].command_length = 3;
  CMD[i].supported = true;
  i = 5;
  CMD[i].command_code = 'C';
  CMD[i].command_length = 3;
  CMD[i].supported = true;
  i = 6;
  CMD[i].command_code = 'X';
  CMD[i].command_length = 3;
  CMD[i].supported = true;
  i = 7;
  CMD[i].command_code = 'I';
  CMD[i].command_length = 2;
  CMD[i].supported = true;
  i = 8;
  CMD[i].command_code = 'V';
  CMD[i].command_length = 2;
  CMD[i].supported = true;
  i = 9;
  CMD[i].command_code = 'Q';
  CMD[i].command_length = 3;
  CMD[i].supported = true;
  i = 10;
  CMD[i].command_code = 'G';
  CMD[i].command_length = 3;
  CMD[i].supported = false;
  i = 11;
  CMD[i].command_code = 'O';
  CMD[i].command_length = 3;
  CMD[i].supported = true;
  i = 12;
  CMD[i].command_code = 'M';
  CMD[i].command_length = 2;
  CMD[i].supported = true;
  i = 13;
  CMD[i].command_code = 'P';
  CMD[i].command_length = 3;
  CMD[i].supported = true;
  i = 14;
  CMD[i].command_code = 'R';
  CMD[i].command_length = 3;
  CMD[i].supported = true;
  i = 15;
  CMD[i].command_code = 'A';
  CMD[i].command_length = 2;
  CMD[i].supported = true;

  //16 commands in total
}


//Commands can be received from the debugger (Serial) or BLE.  This section is for the Serial input
void SCI_check() {

  if (Serial.available() > 0) {
    // Only do something if there is data in the SCI receive buffer
    if (SCI_in()) {
      Decode_CMD();  //Execute_CMD();
    }
  }
}


boolean SCI_in() {

  delay(100);       //wait for all the characters to arrive at 9600 Baud (~1ms per character)
  char buffer[10];  //buffer for incoming messages

  byte message_length = Serial.available();
  bool message_valid = false;
  bool length_valid = false;


  //Read the first byte in the queue--it should be the command code
  Command = Serial.read();

  message_valid = Validate_command_code(Command);
  length_valid = Validate_command_length(message_length);

  //If command found in the list and enough characters, continue.  Otherwise, return false and purge the input queue
  if (!(message_valid && length_valid)) {
    while (Serial.available() != 0) Serial.read();  //For this and other Serial.x, use SerialBT for ESP32
    return false;
  }

  //At this point, the Command code has been validated and the rest of command is still in the SerialBT input queue
  //Next, we read the rest of the command and convert to Hex data.

  if (Command_length == 2 || Command_length == 3) {
    buffer[0] = Serial.read();
    buffer[1] = Serial.read();
    Parameter_1 = get_Parameter(buffer[0], buffer[1]);
    //Serial.println(Parameter_1);
  }

  if (Command_length == 3) {
    buffer[0] = Serial.read();
    buffer[1] = Serial.read();
    Parameter_2 = get_Parameter(buffer[0], buffer[1]);
    //Serial.println(Parameter_2);
  }

  //At this point, the Command code and Parameters are set and we are done with the Serial port for now
  //  Now decode the command code and subcode and action code that are in Parameters
  if (message_valid) return true;
  else
    return false;
}


//Validate the command by looking it up in the CMD list, and get the length
bool Validate_command_code(char Command_code) {
  for (int i = 0; i < number_of_commands; i++) {
    if (Command == CMD[i].command_code && CMD[i].supported == true) {
      Command_length = CMD[i].command_length;
      return true;
    }
  }
  return false;
}

//make sure there are enough characters in the queue for the command
bool Validate_command_length(byte message_length) {
  if (message_length + 1 < 2 * Command_length) return false;
  else
    return true;
}



//This support routine converts the two bytes of ASCII data into a HEX number from 0 to FF.
uint8_t get_Parameter(char msb, char lsb) {
  uint8_t hex1, hex2;
  uint8_t Parameter;

  msb = toupper(msb);
  lsb = toupper(lsb);

  if ((msb - '0') <= 9) hex1 = msb - '0';
  else
    hex1 = msb - 'A' + 10;
  if ((lsb - '0') <= 9) hex2 = lsb - '0';
  else
    hex2 = lsb - 'A' + 10;
  Parameter = (16 * hex1) + hex2;
  return Parameter;
}

/*  This routine uses the Command and Parameters to determine which HCI field is getting updated and
     to determine whether the action is to increment/decrement the value or jump to a specific value

     The output of this routine is the current_field_ID, action_code, and response string that gets sent back to the client
*/
void Decode_CMD() {
  char txString[16];
  String response = "";
  if (Command_length == 2) response = Decode_2byte_Commands(Command, Parameter_1);
  if (Command_length == 3) response = Decode_3byte_Commands(Command, Parameter_1, Parameter_2);

  //send response back to BLE master
  for (byte i = 0; i < (response.length() + 1); i++) txString[i] = response[i];

  Serial.print("response = ");
  Serial.println(txString);

  pCharacteristic->setValue(txString);
  pCharacteristic->notify();  // Send the value to the app!

  //SerialBT.print(response);        //send back the acknowledgement with the HCI value to display
  //Serial.println(response);       //debug
}

String Decode_2byte_Commands(char Command, uint8_t Parameter_1) {
  String response;

  switch (Command) {

    case 'I':
      //Command = "Input_select"
      /* 0:  Analog
         1:  WiFi Streamer (SPDIF1)
         2:  PC (SPDIF2)
         3:  Test
      */

      current_field_ID = Input;
      response = update_field("I", current_field_ID, Parameter_1);
      Serial.print(Parameter_1);

      response = "I00" + String(Parameter_1);     //overwrite response to BLE app
      break;

    case 'V':
      //Command = "ADAU1466 Mute"
      //Mute codes in Parameter_1:  0 = mute    1 = unmute
      if (Parameter_1 == 0) {
        //initialization_mute();
        response = "VMute";  //send back an acknowledgement
      } else {
        //initialization_unmute();
        response = "VPlay";  //send back an acknowledgement
      }
      break;

    case 'Z':
      //Command = "ADAU1466 Main Volume"
      //Volume is an "HCI index" sent as a Hex string, converted to a byte, in Parameter_1
      //          ADAU1466_main_vol();
      current_field_ID = Main_volume;
      response = update_field("Z", current_field_ID, Parameter_1);
      break;


    case 'S':
      //Command = "Save/Restore"
      //Save codes in Parameter_1:  0 = save    1 = save to alternate block 1    2 = restore from alternate block

      switch (Parameter_1) {
        case 0:
          //Debug:  disable save_state
          save_state();  //save the HCI state in EEPROM
          response = "S0";
          break;
        case 1:
          //dont_save_state--just send back a response
          response = "S1";
          break;
        case 2:
          save_state_a();
          response = "S2";
          break;
        case 3:
          restore_state_a();
          response = "S3";
          break;
        default:
          break;
      }
      break;

    case 'A':
      //Command = "Speaker_Select"
      /*Select codes in Parameter_1: 
     0 = save    1 = save to alternate block 1    2 = restore from alternate block
     */
      current_field_ID = Speaker_select;
      response = update_field("A", current_field_ID, Parameter_1);
      break;
  }
  return response;
}

String Decode_3byte_Commands(char Command, uint8_t Parameter_1, uint8_t Parameter_2) {
  String response = "";

  switch (Command) {
    case 'B':
      //Command = "BSC Spec"
      //BSC Frequency code in Parameter_1;  Gain in Parameter_2
      //BSC subcodes:   0 =  BSC_freq
      //                1 =  BSC_Gain
      response = "B";
      switch (Parameter_1) {
        case 0:
          current_field_ID = BSC_freq;
          response.concat('0');  //Response subcode
          break;
        case 1:
          current_field_ID = BSC_Gain;
          response.concat('1');  //Response subcode
          break;
        default: current_field_ID = Main_volume;
      }

      response = update_field(response, current_field_ID, Parameter_2);
      break;

    case 'C':
      //Command = "ADAU1466 Volume Trim"
      //Channel number in Parameter_1;  8-bit volume trim is in Parameter_2
      //Channel codes:  0 = SuperTweeter  (all channels are stereo)
      //                1 = Tweeter
      //                2 = Midrange
      //                3 = Woofer
      //                4 = Sub


      response = "C";  //First part of response is the echoed command code
      switch (Parameter_1) {
        case 0:
          current_field_ID = chan_vol_T;
          response.concat('0');  //Response subcode
          break;
        case 1:
          current_field_ID = chan_vol_W;
          response.concat('1');  //Response subcode
          break;
        case 2:
          current_field_ID = chan_vol_S;
          response.concat('2');  //Response subcode
          break;
        default: current_field_ID = Main_volume;
      }
      //common ending for all cases:  update the value and grab the response string

      response = update_field(response, current_field_ID, Parameter_2);
      break;

    case 'X':
      //Command = "ADAU1466 Delay"
      //Channel number is in Parameter_1;  Delay value is in Parameter_2
      //          ADAU1466_delay();

      response = "X";

      switch (Parameter_1) {
        case 0:
          current_field_ID = Delay_T;
          response.concat('0');  //Response subcode
          break;

        case 1:
          current_field_ID = Delay_W;
          response.concat('1');  //Response subcode
          break;

        case 2:
          current_field_ID = Delay_S;
          response.concat('2');  //Response subcode
          break;

        default: current_field_ID = Main_volume;
      }
      //common ending for all cases:  update the value and grab the response string
      response = update_field(response, current_field_ID, Parameter_2);
      break;


    case 'F':
      //Command = "Tweeter-Woofer Crossover Spec"
      //Crossover field code in Parameter_1;  value in Parameter_2
      //Crossover field codes:  0 =  Xover_TW_type
      //                        1 =  Xover_TW_freq
      //                        2 =  Xover_WS_type
      //                        3 =  Xover_WS_freq

      response = "F";  //First part of response is the echoed command code

      switch (Parameter_1) {
        case 0:
          current_field_ID = Xover_T_W_type;
          response.concat('0');  //Response subcode
          break;
        case 1:
          current_field_ID = Xover_T_W_freq;
          response.concat('1');  //Response subcode
          break;
        case 2:
          current_field_ID = Xover_W_S_type;
          response.concat('2');  //Response subcode
          break;
        case 3:
          current_field_ID = Xover_W_S_freq;
          response.concat('3');  //Response subcode
          break;

        case 8:
          current_field_ID = Xover_T_W_polarity;
          response.concat('8');  //Response subcode
          break;
        case 9:
          current_field_ID = Xover_W_S_polarity;
          response.concat('9');  //Response subcode
          break;

        default: current_field_ID = Main_volume;
      }
      //common ending for all cases:  update the value and grab the response string
      response = update_field(response, current_field_ID, Parameter_2);
      break;

    case 'E':
      //Command = "EQ"
      //EQ Band in Parameter_1;  EQ Level in Parameter_2

      response = "E";  //First part of response is the echoed command code

      switch (Parameter_1) {
        case 0:
          current_field_ID = EQ_32_Gain;
          response.concat('0');  //Response subcode
          break;
        case 1:
          current_field_ID = EQ_64_Gain;
          response.concat('1');  //Response subcode
          break;
        case 2:
          current_field_ID = EQ_125_Gain;
          response.concat('2');  //Response subcode
          break;
        case 3:
          current_field_ID = EQ_250_Gain;
          response.concat('3');  //Response subcode
          break;
        case 4:
          current_field_ID = EQ_500_Gain;
          response.concat('4');  //Response subcode
          break;
        case 5:
          current_field_ID = EQ_1K_Gain;
          response.concat('5');  //Response subcode
          break;
        case 6:
          current_field_ID = EQ_2K_Gain;
          response.concat('6');  //Response subcode
          break;
        case 7:
          current_field_ID = EQ_4K_Gain;
          response.concat('7');  //Response subcode
          break;
        case 8:
          current_field_ID = EQ_8K_Gain;
          response.concat('8');  //Response subcode
          break;
        case 9:
          current_field_ID = EQ_16K_Gain;
          response.concat('9');  //Response subcode
          break;
        case 10:
          current_field_ID = EQ_reset_Gain;
          response.concat('A');  //Response subcode
          break;
        default: current_field_ID = Main_volume;
      }

      if ((Parameter_2 >= 253) || (Parameter_2 <= EQ_gain_qty)) {
        current_value(current_field_ID, Parameter_2, true);  //execute the EQ update
      }
      response.concat(field_string_global);  //Next part of the response is the EQ string
      break;

    case 'K':
      /*Command = "Bass Enhancement"
        Sub-command in Parameter_1;  value in Parameter_2
        Sub-commands:      0 = Rumble Freq
                           1 = Rumble Q
                           2 = Peak Freq
                           3 = Peak Gain
                           4 = Bass Enh Mode
                           5 = SuperBass_freq
                           6 = SuperBass_Intensity
                           7 = SuperBass_Gain
      */

      response = "K";  //First part of response is the echoed command code

      switch (Parameter_1) {
        case 0:
          current_field_ID = Rumble_filter_freq;
          response.concat('0');  //Response subcode
          break;
        case 1:
          current_field_ID = Rumble_filter_Q;
          response.concat('1');  //Response subcode
          break;
        case 2:
          current_field_ID = Peak_filter_freq;
          response.concat('2');  //Response subcode
          break;
        case 3:
          current_field_ID = Peak_filter_Gain;
          response.concat('3');  //Response subcode
          break;
        case 4:
          current_field_ID = Bass_Enhancement_Mode;
          response.concat('4');  //Response subcode
          break;
      }
      response = update_field(response, current_field_ID, Parameter_2);
      break;

    case 'Q':

      /*Command = "Custom Filter"
          Sub-command in Parameter_1;  value in Parameter_2
        Sub-commands:     0 = Shared1_Type
                          1 = Shared1_Freq
                          2 = Shared1_Q
                          3 = Shared1_Gain
                          4 = Tweeter_Type
                          5 = Tweeter_Freq
                          6 = Tweeter_Q
                          7 = Tweeter_Gain
                          8 = Woofer_Type
                          9 = Woofer_Freq
                          10 = Woofer_Q
                          11 = Woofer_Gain
                          12 = Sub_Type
                          13 = Sub_Freq
                          14 = Sub_Q
                          15 = Sub_Gain
                          16 = Clear All Filters
      */
      response = "Q";  //First part of response is the echoed command code

      switch (Parameter_1) {
        case 0:
          current_field_ID = Custom_Shared1_Type;
          response.concat('0');  //Response subcode
          break;
        case 1:
          current_field_ID = Custom_Shared1_Freq;
          response.concat('1');  //Response subcode
          break;
        case 2:
          current_field_ID = Custom_Shared1_Q;
          response.concat('2');  //Response subcode
          break;
        case 3:
          current_field_ID = Custom_Shared1_Gain;
          response.concat('3');  //Response subcode
          break;
        case 4:
          current_field_ID = Custom_Tweeter_Type;
          response.concat('4');  //Response subcode
          break;
        case 5:
          current_field_ID = Custom_Tweeter_Freq;
          response.concat('5');  //Response subcode
          break;
        case 6:
          current_field_ID = Custom_Tweeter_Q;
          response.concat('6');  //Response subcode
          break;
        case 7:
          current_field_ID = Custom_Tweeter_Gain;
          response.concat('7');  //Response subcode
          break;
        case 8:
          current_field_ID = Custom_Woofer_Type;
          response.concat('8');  //Response subcode
          break;
        case 9:
          current_field_ID = Custom_Woofer_Freq;
          response.concat('9');  //Response subcode
          break;
        case 10:
          current_field_ID = Custom_Woofer_Q;
          response.concat('A');  //Response subcode
          break;
        case 11:
          current_field_ID = Custom_Woofer_Gain;
          response.concat('B');  //Response subcode
          break;
        case 12:
          current_field_ID = Custom_Sub_Type;
          response.concat('C');  //Response subcode
          break;
        case 13:
          current_field_ID = Custom_Sub_Freq;
          response.concat('D');  //Response subcode
          break;
        case 14:
          current_field_ID = Custom_Sub_Q;
          response.concat('E');  //Response subcode
          break;
        case 15:
          current_field_ID = Custom_Sub_Gain;
          response.concat('F');  //Response subcode
          break;

        case 16:
          current_field_ID = Custom_Reset;
          response.concat('G');  //Response subcode
          break;
      }
      response = update_field(response, current_field_ID, Parameter_2);
      break;

    case 'O':

      /*Command = "ADAU1466 Source Mode"
          Sub-command in Parameter_1;  value in Parameter_2
        Sub-commands:     0 = Source Select
                          1 = Sine Wave Frequency
                          2 = Sweep Range
                          3 = Sweep Steps
                          4 = Sweep Speed
      */
      response = "O";  //First part of response is the echoed command code

      switch (Parameter_1) {

        case 0:
          current_field_ID = ADAU1466_Source;
          response.concat('0');  //Response subcode
          break;
        case 1:
          current_field_ID = ADAU1466_Freq;
          response.concat('1');  //Response subcode
          break;
        case 2:
          current_field_ID = ADAU1466_Range;
          response.concat('2');  //Response subcode
          break;
        case 3:
          current_field_ID = ADAU1466_Steps;
          response.concat('3');  //Response subcode
          break;
        case 4:
          current_field_ID = ADAU1466_Speed;
          response.concat('4');  //Response subcode
          break;

        default:
          break;
      }
      response = update_field(response, current_field_ID, Parameter_2);
      break;

    case 'P':

      /*Command = "Curvature"
        Configuration in Parameter_1;  value in Parameter_2
        Configurations:   0 = 12 unique tweeters & 7 unique woofers
                          1-3 = TBD

      */
      response = "P";  //First part of response is the echoed command code

      switch (Parameter_1) {

        case 0:
          current_field_ID = Curvature;
          response.concat('0');  //Response subcode/configuration
          break;
        case 1:
          //place-holder
          response.concat('1');  //Response subcode
          break;

        default:
          break;
      }
      response = update_field(response, current_field_ID, Parameter_2);
      break;

    case 'R':

      /*Command = "Curvature"
        Configuration in Parameter_1;  value in Parameter_2
        Configurations:   0 = 12 unique tweeters & 7 unique woofers
                          1-3 = TBD

      */
      response = "R";  //First part of response is the echoed command code

      switch (Parameter_1) {

        case 0:
          current_field_ID = Shading;
          response.concat('0');  //Response subcode/configuration
          break;
        case 1:
          //place-holder
          response.concat('1');  //Response subcode
          break;

        default:
          break;
      }
      response = update_field(response, current_field_ID, Parameter_2);
      break;
  }

  return response;
}

String update_field(String CMD_code, byte field_ID, uint8_t Parameter) {
  String response = CMD_code;
  current_value(field_ID, Parameter, true);
  response.concat(field_string_global);
  return response;
}

/****  This code is for the BLE Input   ****
*/

void BLE_RX_check() {
  // Only do something if there is data in the SCI receive buffer
  if (BLE_in()) {
    Decode_CMD();
  }
}

bool BLE_in() {
  boolean message_valid = false;
  boolean length_valid = false;

  extern String rxValue;

  char buffer[10];  //buffer for incoming messages

  if (rxValue.length() > 0) {
    Serial.print("BLE Command: ");
    for (int i = 0; i < rxValue.length(); i++) {
      Serial.print(rxValue[i]);
    }

    //Data in BLE read buffer
    Command = rxValue[0];
    message_valid = Validate_command_code(Command);

    byte message_length = rxValue.length();
    length_valid = Validate_command_length(message_length);

    //If command found in the list and enough characters, continue.  Otherwise, return false and purge the input queue
    if (!(message_valid && length_valid)) {
      rx_flag = false;
      return false;
    }

    Serial.println("Message validated");

    //At this point, the Command code has been validated and the rest of command is still in the SerialBT input queue
    //Next, we read the rest of the command and convert to Hex data.

    //send_command_to_SPDIF(rxValue);         //first, send a copy to the other speaker via SPDIF user data
    //tx_timer = millis() + TX_revisit_time;  //set timer to send out the zero command from the main loop

    if (Command_length == 2 || Command_length == 3) {
      Parameter_1 = get_Parameter(rxValue[1], rxValue[2]);
      //Serial.println(Parameter_1);
    }

    if (Command_length == 3) {
      Parameter_2 = get_Parameter(rxValue[3], rxValue[4]);
      //Serial.println(Parameter_2);
    }
    rx_flag = false;
    return true;
  }
  return false;  //should never get here but compiler squawks if all return possibilities aren't defined
}

void send_command_to_SPDIF(String BLE_received) {
  /*
  This rotine is called when a BLE command is received and validated.  It copies the command to the SPDIF transmitter where it is sent to the other speaker.
  The command is allowed to sit in the user data for at least two 192-bit SPDIF frames (10ms) to ensure that the other speaker "sees" it, and then the command is zeroed out.
  The zeroed command is allowed to stay in the user data for at least two frames, at which point the other speaker will be ready to accept another command.
  Since all of this is associated with a "slow" BLE command, there is no harm in using the blocking delay() function.
  This code is for the transmit side.
  */

  uint32_t reg_read_val;
  uint8_t addr_MSB = 0xF6;
  uint8_t addr_LSB = 0xC0;  //left channel user data TX register address

  Serial.print("BLE_received = ");
  Serial.println(BLE_received);

  Lib_SPI.SPI_write_reg_string(addr_MSB, addr_LSB, BLE_received);  //copy the command to the SPDIF transmitter

  for (int i = 0; i < BLE_received.length(); i++) {               //Read back from SPDIF transmitter to verify that user data was written
    reg_read_val = Lib_SPI.SPI_read_reg(addr_MSB, addr_LSB + i);  //Reg address
    //Serial.print("TX data = ");
    //Serial.println((char)reg_read_val);
  }

  //Lib_SPI.SPI_write_reg_string(0xF6, 0x9F, (String)0);  //select registers as source for AUX bits
}

void send_ID_and_value_to_SPDIF(uint8_t ID, uint8_t value) {
  uint32_t reg_read_val;
  uint8_t addr_MSB = 0xF6;
  uint8_t addr_LSB = 0xC0;  //left channel user data TX register address

  Lib_SPI.SPI_write_reg_int(addr_MSB, addr_LSB, ID);         //copy the command to the SPDIF transmitter
  Lib_SPI.SPI_write_reg_int(addr_MSB, addr_LSB + 1, value);  //copy the command to the SPDIF transmitter
}

void SPDIF_TX_clear() {
  uint32_t reg_read_val;
  uint8_t addr_MSB = 0xF6;
  uint8_t addr_LSB = 0xC0;  //left channel user data TX register address

  Lib_SPI.SPI_write_reg_int(addr_MSB, addr_LSB, 0);
  //Serial.print("TX_clear = ");
  Serial.println(Lib_SPI.SPI_read_reg(addr_MSB, addr_LSB), HEX);
}


void SPDIF_check() {

  uint32_t reg_read_val;
  uint32_t field_ID;
  uint32_t value;
  /*
  This is  for receiving SPDIF messages (User Bits)
  There are 4 states:
    0.  Waiting for the first byte of the command to change from 0 to non-zero
    1.  Got some command data but it might not be valid.  Wait for 10ms timer to expire to check again
    2.  Got the second command, which should be valid.
    3.  Waiting for the command to be zero. After the command is zero, execute the command and then return to state 1

  On the transmit side, the transmitter puts the field_ID/value pair data into the transmit buffer and then sets a 50ms timer
  After 50ms elapses, the transmitter sends a "zero" command.  The transmit code is at the end of the HCI.ino current_value
  routine, where the serial commands have been converted to name/value pairs.
  */

  reg_read_val = Lib_SPI.SPI_read_reg(0xF6, 0x30);
  switch (SPDIF_RX_state) {
    case 0:
      //Waiting for first byte of the command to change from 0 to non-zero

      if (!((reg_read_val == '0') || (reg_read_val == 0))) {
        SPDIF_RX_state = 1;
      }
      //Serial.print("SPDIF_RX_state = ");
      //Serial.println(SPDIF_RX_state);
      //otherwise, stay in state 0
      break;

    case 1:
      //something arrived in the User Data, but it might be corrupt, so wait 10ms
      SPDIF_RX_state = 2;
      break;

    case 2:
      //It is now about 10ms after something first showed up in the User Data.  By now, the data should be stable
      //Now, assemble the command from the User Data bytes
      SPDIF_RX_state = 3;
      //get_CMD_and_parameters();  //call the routine that extracts the command and the parameters from the registers
      field_ID = Lib_SPI.SPI_read_reg(0xF6, 0x30);  //read first byte of user data, value);
      Serial.print("field_ID ");
      Serial.println(field_ID);

      value = Lib_SPI.SPI_read_reg(0xF6, 0x31);  //read second byte of user data, value);
      Serial.print("value ");
      Serial.println(value);
      break;

    case 3:
      //The command has been received and we need to wait until the command is zeroed out by the transmitter.
      //Then, execute the command
      if ((reg_read_val == '0') || (reg_read_val == 0)) {
        SPDIF_RX_state = 0;
        //update_from_HCI(field_ID, value);   //this executes the ID with the right value
      }
      //otherwise, stay in state 3, waiting for a zero in the first User Data register
      break;

    default:
      break;
  }
  //common ending for all states:  update the timer
  rx_timer = millis() + RX_revisit_time;
}
