/*
  ADAU1466_cmd.cpp - Library for sending DSP commands to the ADAU1466
  These routines convert the HCI values to parameters that are used by the DSP functions.

  The input is the HCI code and the associated Parameter RAM address. Some of the commands are "higher level"
  than others, in that they call related ADAU1466_DSP functions to do the lower level work, whereas some (e.g. Volume) call the
  lower level functions in ADAU1466_I2C directly.  This disparity in abstraction will get addressed in future revisions.

  Created by Neil Davis, December, 2020.
*/

#include "Arduino.h"
#include "ADAU1466_cmd.h"
#include "ADAU1466_Lookup_tables.h"
#include <ADAU1466_SPI.h>
#include "ADAU1466_DSP.h"
#include "Cell_map.h"
#include <ADAU1466_Shared_enums.h>

extern ADAU1466_SPI Lib_SPI;
extern bool setup_done;

const uint8_t ADAU1466_addr = 0x3B;
uint8_t Cmd_Data[5];
int delay_amount;
const int delay_HF = 10;
const int delay_Mid = 20;
const int delay_LF = 200;
const int delay_none = 1;

const int led = 25;

//ADAU1466_I2C my_SPI(ADAU1466_addr, SafeLoadModule);  //for this library assume a fixed address at 3B

ADAU1466_DSP myADAU1466_DSP;

ADAU1466_cmd::ADAU1466_cmd(word _Mute_addresses[number_of_mute_addresses]) {
  for (int i = 0; i < number_of_mute_addresses; i++) {
    _copy_of_Mute_addresses[i] = _Mute_addresses[i];
    //Serial.print("Mute_addr:  ");
    //Serial.println(_Mute_addresses[i]);
  }
}

void ADAU1466_cmd::Main_Volume(int reg_num, int value_code) {
  Lib_SPI.safeload_addr(reg_num);                   //set up safeload address
  convert_Vol_to_8_24(Main_Vol_table, value_code);  //look up the volume level and send it
}

void ADAU1466_cmd::Chan_vol_trim(int reg_num, int value_code) {
  Lib_SPI.safeload_addr(reg_num);                      //set up safeload address
  convert_Vol_to_8_24(Trim_Volume_table, value_code);  //look up the volume level and send it
}

void ADAU1466_cmd::Update_Delay(int reg_num, int delay_level) {
  Lib_SPI.safeload_addr(reg_num);  //set up safeload address
  convert_delay_level(delay_level);
}


/* The overall flow for the crossover calculation is:
    1)  Convert the HCI code to frequencies and filter types (convert_freq_code and convert_Xover_code)
    2)  Look up the filter type to determine the number of biquads needed and their parameters (get_biquad_spec)
        The biquad_spec has the following information:
           --filter frequency
           --the number of biquads needed to implement the crossover
           --the filter order of each biquad
           --the Q of each biquad
    3)  For each biquad, Calculate the coefficients and send them to the ADAU1466 (crossover_calc)
        This is done in two routines:
          a) biquad_load, which puts the coefficient data into the biquad Parameter RAM
          a) fill_TW_with_none and fill_WS_with_none, which puts a "1" (passthrough) in the unused biquads for each crossover
*/

void ADAU1466_cmd::Input_select(int reg_num, int value_code) {
  /* These are the Input codes for the SSM3582/ADAU1466 board:
      0:  Bluetooth
      1:  Analog (PCM1808)
      2:  SPDIF (WiFi module)
      3:  ADAU1466-generated sources
  */
  uint8_t Cmd_Data[4];

  Cmd_Data[0] = 0;
  Cmd_Data[1] = 0;
  Cmd_Data[2] = 0;
  Cmd_Data[3] = 0;

  //Serial.print("input_code ");
  //Serial.println(value_code);

  switch (value_code) {
    case 0:
      Cmd_Data[3] = 0x0;
      break;
    case 1:
      Cmd_Data[3] = 0x2;
      break;
    case 2:
      Cmd_Data[3] = 0x4;
      break;
    case 3:
      Cmd_Data[3] = 0x6;  //not used in Act 2
      break;
  }

  Lib_SPI.safeload_addr(reg_num);   //set up safeload address
  Lib_SPI.safeload_data(Cmd_Data);  //send data to safeload registers
  Lib_SPI.safeload_xfer(1);
  delay(2);  //wait for next audio sample
}

void ADAU1466_cmd::T_W_Xover(byte freq_code, byte Xover_type_code, word biquad_addresses[5][2]) {
  Shared_enum::Xover_drivers driver_set = Shared_enum::T_W;
  float FC = myADAU1466_DSP.convert_freq_code(driver_set, freq_code);                           //find the floating point frequency value
  Shared_enum::Xovers x_type = myADAU1466_DSP.convert_Xover_code(driver_set, Xover_type_code);  //convert Xover to an enumerated type
  Shared_enum::BQ_spec biquads = myADAU1466_DSP.get_biquad_spec(FC, x_type);                    //look up the number of biquads and their parameters.  Fill in the biquad_spec structure
  Mute();                                                                                       //might be major changes, so set the mute flag on
  myADAU1466_DSP.BQ5_Xover_calc(biquads, biquad_addresses);                                     //calculate the biquads and load them into the ADAU1466
  delay(delay_Mid);                                                                             //allow filter to settle
  Unmute();
}

void ADAU1466_cmd::W_S_Xover(byte freq_code, byte Xover_type_code, word biquad_addresses[4][2]) {
  Shared_enum::Xover_drivers driver_set = Shared_enum::W_S;
  float FC = myADAU1466_DSP.convert_freq_code(driver_set, freq_code);                           //find the floating point frequency value
  Shared_enum::Xovers x_type = myADAU1466_DSP.convert_Xover_code(driver_set, Xover_type_code);  //convert to an enumerated type
  Shared_enum::BQ_spec biquads = myADAU1466_DSP.get_biquad_spec(FC, x_type);                    //look up the number of biquads and their parameters.  Fill in the biquad_spec structure
  Mute();                                                                                       //might be major changes, so set the mute flag on
  myADAU1466_DSP.BQ3_Xover_calc(biquads, biquad_addresses);                                     //calculate the biquads and load them into the ADAU1466
  delay(delay_LF);                                                                              //allow filter to settle
  Unmute();
}

void ADAU1466_cmd::BSC_Freq_or_Gain(int reg_num, int freq_code, int gain_code) {
  Shared_enum::Filter_spec BSC_filter;

  BSC_filter.Q = .707;
  BSC_filter.filter_type = Shared_enum::LP_Shelf;
  BSC_filter.Freq_center = Convert_BSC_freq_code(freq_code);  //find the floating point frequency value
  BSC_filter.Gain = Convert_BSC_gain_code(gain_code);         //find the floating point gain value
  //BSC_filter.biquad_addr = reg_num;

  Mute();
  //filter_calc();
  myADAU1466_DSP.filter_calc(BSC_filter, reg_num);
  delay(50);  //allow filter to settle
  Unmute();
}


void ADAU1466_cmd::Xover_T_polarity(int T_Pol_Addr, int value_code) {
  switch (value_code) {
    case 0:
      Set_invert_to_off(T_Pol_Addr);
      break;
    case 1:
      Set_invert_to_on(T_Pol_Addr);
      break;
    default:
      Set_invert_to_off(T_Pol_Addr);
  }
}

void ADAU1466_cmd::Xover_W_polarity(int W_Pol_Addr, int value_code) {
  switch (value_code) {
    case 0:
      Set_invert_to_off(W_Pol_Addr);
      break;
    case 1:
      Set_invert_to_on(W_Pol_Addr);
      break;
    default:
      Set_invert_to_off(W_Pol_Addr);
  }
}

void ADAU1466_cmd::Xover_S_polarity(int S_Pol_Addr, int value_code) {
  switch (value_code) {
    case 0:
      Set_invert_to_off(S_Pol_Addr);
      break;
    case 1:
      Set_invert_to_on(S_Pol_Addr);
      break;
    default:
      Set_invert_to_off(S_Pol_Addr);
  }
}

void ADAU1466_cmd::EQ_Filter(int reg_num, float Freq, int value_code) {
  Shared_enum::Filter_spec EQ_filter;

  EQ_filter.Q = .707;
  EQ_filter.filter_type = Shared_enum::Peaking;
  EQ_filter.Freq_center = Freq;
  EQ_filter.Gain = Convert_EQ_gain_code(value_code);  //find the filter gain

  /*Serial.print("Gain = ");
  Serial.println(EQ_filter.Gain);
  Serial.print("Freq = ");
  Serial.println(EQ_filter.Freq_center);
  Serial.print("Q = ");
  Serial.println(EQ_filter.Q);
  Serial.print("Reg = ");
  Serial.println(reg_num);
  */

  Mute();
  myADAU1466_DSP.filter_calc(EQ_filter, reg_num);
  delay(50);  //allow filter to settle
  Unmute();
}

void ADAU1466_cmd::Rumble_filter(word Rumble_addr, int freq_code, int Q_code, bool Rumble_enabled) {
  Shared_enum::Filter_spec Rumble;
  Rumble.Freq_center = Bass_Freq_table[freq_code];
  Rumble.Q = Bass_Q_table[Q_code];
  Rumble.Gain = 1.0;
  if (Rumble_enabled == true) {
    Rumble.filter_type = Shared_enum::Two_Pole_HP;
  } else {
    Rumble.filter_type = Shared_enum::None;
  }

  /*
    Serial.print("Rumble enum:  ");
    Serial.print(Rumble.filter_type);
    Serial.print("  enabled: ");
    Serial.println(Rumble_enabled);
    Serial.print("  Freq: ");
    Serial.print(Rumble.Freq_center);
    Serial.print("  Q: ");
    Serial.println(Rumble.Q);
  */

  Mute();
  myADAU1466_DSP.filter_calc(Rumble, Rumble_biquad);  //rumble filter enum and rumble address
  delay(200);                                         //allow filter to settle
  Unmute();
}

void ADAU1466_cmd::Peak_filter(word Peak_biquad, int freq_code, int gain_code, bool Peak_enabled) {
  Shared_enum::Filter_spec Peak;
  Peak.Freq_center = Bass_Freq_table[freq_code];
  Peak.Q = 1.0;
  Peak.Gain = Bass_Gain_table[gain_code];
  if (Peak_enabled == true) {
    Peak.filter_type = Shared_enum::Peaking;
  } else {
    Peak.filter_type = Shared_enum::None;
  }

  Mute();
  myADAU1466_DSP.filter_calc(Peak, Peak_biquad);  //Peak is filter enum;  Peak_filter is the I2C address
  delay(200);                                     //allow filter to settle
  Unmute();
}


void ADAU1466_cmd::Custom_filter(int reg_num, Shared_enum::tfilters filter_type, float filter_freq, float filter_gain, float filter_Q) {
  Shared_enum::Filter_spec Custom;
  Custom.filter_type = filter_type;
  Custom.Freq_center = filter_freq;
  Custom.Gain = filter_gain;
  Custom.Q = filter_Q;

  Mute();
  myADAU1466_DSP.filter_calc(Custom, reg_num);
  delay(10);                          //allow filter to settle
  if (filter_freq < 1000) delay(50);  //if a low frequency, delay longer
  if (filter_freq < 100) delay(200);
  Unmute();
}


void ADAU1466_cmd::ADAU1466_Source_select(int reg_num, byte source_code, byte sweep_reg) {
  /* These are the Input codes ADAU1466 sources:
        /* These are the Input codes for the SSM3582/ADAU1466 board:
      0:  Sine wave (Tone)
      1:  Sweep
      2:  White noise
      3:  VCO (controlled by the volume control on the ADAU1466/SSM3852 board)
      4:  Mute

      The Source Select command is also used to trigger the Sweep.  If Sweep is selected, it will toggle the Sweep On/Off function
  */
  uint8_t Cmd_Data[4];

  Cmd_Data[0] = 0;
  Cmd_Data[1] = 0;
  Cmd_Data[2] = 0;
  Cmd_Data[3] = 0;

  //Serial.print("source_code ");
  //Serial.println(source_code);

  switch (source_code) {
    case 0:
      Cmd_Data[3] = 0x0;
      break;
    case 1:
      Cmd_Data[3] = 0x1;
      break;
    case 2:
      Cmd_Data[3] = 0x2;
      break;
    case 3:
      Cmd_Data[3] = 0x3;
      break;
    case 4:
      Cmd_Data[3] = 0x4;
      break;
  }

  Lib_SPI.safeload_addr(reg_num);   //set up safeload address
  Lib_SPI.safeload_data(Cmd_Data);  //send volume data to safeload registers
  Lib_SPI.safeload_xfer(1);
  delay(2);  //wait for next audio sample

  if (source_code == 1) {  //check for Sweep button
    Cmd_Data[0] = 0;
    Cmd_Data[1] = 0;
    Cmd_Data[2] = 0;
    Cmd_Data[3] = 0;                       //turn the on/off switch at offset = 5 to "off"
    Lib_SPI.safeload_addr(sweep_reg + 5);  //set up safeload address
    Lib_SPI.safeload_data(Cmd_Data);       //send volume data to safeload registers
    Lib_SPI.safeload_xfer(1);
    Cmd_Data[0] = 1;                       //turn the switch on
    Lib_SPI.safeload_addr(sweep_reg + 5);  //set up safeload address
    Lib_SPI.safeload_data(Cmd_Data);       //send volume data to safeload registers
    Lib_SPI.safeload_xfer(1);
    delay(1);  //wait for next audio sample
  }
}

void ADAU1466_cmd::ADAU1466_Sine(int reg_num, byte freq_code) {
  /* The Sine HCI allows selecting up to 10 frequencies that are multiples of powers of 2, starting at 31.25Hz
    Example:  31.25 * 2^0 = 31.25;   31.25 * 2^1 = 62.5;   31.25 * 2^2 = 125;  etc.
  */

  uint32_t Sine_freq = 0x00005555;           //31.25  (actually 31.25/24000 in 8.24 format)
  Sine_freq = Sine_freq * (2 << freq_code);  //multiply by power of 2
  Sine_freq = Sine_freq / 2;                 //adjust for 2X sampling rate

  uint8_t Cmd_Data[4];
  Cmd_Data[0] = lowByte((Sine_freq & 0xff000000) >> 24);
  Cmd_Data[1] = lowByte((Sine_freq & 0x00ff0000) >> 16);
  Cmd_Data[2] = lowByte((Sine_freq & 0x0000ff00) >> 8);
  Cmd_Data[3] = lowByte((Sine_freq & 0x000000ff));

  reg_num = reg_num + 5;            //offset of 5 for frequency address
  Lib_SPI.safeload_addr(reg_num);   //set up safeload address
  Lib_SPI.safeload_data(Cmd_Data);  //send volume data to safeload registers
  Lib_SPI.safeload_xfer(1);
  delay(1);  //wait for next audio sample
}

void ADAU1466_cmd::ADAU1466_Sweep(int reg_num, byte range_code, byte steps_code, byte speed_code) {
  /*Offsets for the Sweep1 cell:
  Range min = reg_num + 0
  Range Max = reg_num + 1
  Cycles per sweep = reg_num + 2
  Number of steps = reg_num + 4
  Freq_step = reg_num + 3

  Freq_step is calculated as follows:
    temp1 = log(Range_max/Range_min)
    Freq_step = 10^(temp1/(Steps-1))
    Freq_step_hex = Freq_step * (const float Two_to_the_24th = 16777216.0;)
*/
  float Range_max;
  float Range_min;
  //uint32_t Range_max_8_24;
  //uint32_t Range_min_8_24;
  uint32_t Steps_int32;
  uint32_t Speed_int32;
  const float Two_to_the_24th = 16777216.0;
  const float sample_rate = 24000.0;
  uint8_t Cmd_Data[4];
  int reg_num_Rmin = reg_num;
  int reg_num_Rmax = reg_num + 1;
  int reg_num_cycles = reg_num + 2;
  int reg_num_freq_step = reg_num + 3;
  int reg_num_steps = reg_num + 4;

  switch (range_code) {
    case 0:
      Range_min = 20.0;
      Range_max = 20000.0;
      break;
    case 1:
      Range_min = 10.0;
      Range_max = 1000.0;
      break;
    case 2:
      Range_min = 300.0;
      Range_max = 5000.0;
      break;
    case 3:
      Range_min = 1000.0;
      Range_max = 20000.0;
      break;
  }

  float_to_safeload(reg_num_Rmin, Range_min / sample_rate);  //write normalized Range_min to offset = 0
  float_to_safeload(reg_num_Rmax, Range_max / sample_rate);  //write normalized Range_max to offset = 1

  switch (steps_code) {
    case 0:
      Steps_int32 = 10;
      break;
    case 1:
      Steps_int32 = 100;
      break;
    case 2:
      Steps_int32 = 250;
      break;
    case 3:
      Steps_int32 = 500;
      break;
  }

  uint32_to_safeload(reg_num_steps, Steps_int32);  //write number of steps to offset = 4

  //calculate frequency step size
  float temp = log10(Range_max / Range_min);
  float steps = Steps_int32 - 1;
  float temp2 = temp / steps;
  float Freq_step_size = pow(10, temp2);

  //Serial.print("  Freq_step =  ");
  //Serial.println(Freq_step_size, 5);

  float_to_safeload(reg_num_freq_step, Freq_step_size);  //write frequency step size to offset = 3

  switch (speed_code) {
    case 0:
      Speed_int32 = 10;
      break;
    case 1:
      Speed_int32 = 5;
      break;
    case 2:
      Speed_int32 = 2;
      break;
    case 3:
      Speed_int32 = 1;
      break;
  }
  uint32_to_safeload(reg_num_cycles, Speed_int32);  //write speed to offset = 4
}

void ADAU1466_cmd::Curvature_select(int curvature, word tweeter_delay_addresses[12], word woofer_delay_addresses[7]) {
  //integer curvature ranges from 0 to 8, with a response of -4 to +4.  The delay values tables are in the library file "ADAU1466_Lookup_tables.h"
  //The values in these tables are calculated by the excel file "Line_array_amp1.xlsx"
  //The woofer values are in the LA_Woofer_Delay_table_14, and are based on 5 cap angles for 14 woofer channels that are symmetrical about the centerpoint (7 unique channels)
  //The tweeter values are in the LA_Tweeter_Delay_table_24
  //If the line array uses a different number of woofers or tweeters, you will need to include logic to select the proper table, but for now those tables are hard-coded

  const int number_of_woofers = 7;
  const int number_of_tweeters = 12;
  int table_row = 0;
  long table_offset = 0;
  byte table_data = 0;

  for (int t = 0; t <= number_of_tweeters - 1; t++) {
    //The curvature values range from 0 to 8.  "4" is no curvature.  Positive curvature (convex) has the maxixumn delay on the outer drivers--the ones in the center are the "reference"
    //Negative curvature (concave) has the maxixumn delay on the inner drivers of the array.  The ones one the array top and bottom are the "reference"
    //The delay table only has 5 rows, because the concave and convex values are mirrored.  The first row is for the minimum curvature (=4) and the 5th row is maximun curvature

    if (curvature > 4) {  //convex
      table_row = curvature - 4;
      table_offset = t;
    }
    //For negative curvature values we need to negate the row pointer and use the top/bottom tweeters as the "reference"
    if (curvature <= 4) {  //concave or none
      table_row = 4 - curvature;
      table_offset = number_of_tweeters - 1 - t;
    }

    /*
    Serial.print("Curvature: ");
    Serial.println(curvature);
    Serial.print(" table_row: ");
    Serial.println(table_row);
    Serial.print("table_offset: ");
    Serial.println(table_offset);
*/

    table_data = pgm_read_byte_far(&(LA_Tweeter_Delay_table_24[table_row][table_offset]));
    update_driver_delay(tweeter_delay_addresses[t], table_data);  //update both top and bottom tweeters (they are wired in parallel)
  }

  for (int w = 0; w <= number_of_woofers - 1; w++) {
    //Now do the woofers

    if (curvature > 4) {  //convex
      table_row = curvature - 4;
      table_offset = w;
    }

    if (curvature <= 4) {
      table_row = 4 - curvature;
      table_offset = number_of_woofers - 1 - w;
    }

    table_data = pgm_read_byte_far(&(LA_Woofer_Delay_table_14[table_row][table_offset]));
    update_driver_delay(woofer_delay_addresses[w], table_data);  //update both top and bottom woofers (they are wired in parallel)
  }
}

void ADAU1466_cmd::Shading_select(int vol_taper, word tweeter_shading_addresses[12], word woofer_shading_addresses[7]) {  //note:  volume taper = shading (words used interchangeably)
                                                                                                                          //integer curvature ranges from -4 to 4.  The delay values tables are in the library file "ADAU1466_Lookup_tables.h"
  //The values in these tables are calculated by the excel file "Line_array_amp1.xlsx"
  //The woofer values are in the LA_Woofer_Delay_table_14, and are based on 5 cap angles for 14 woofer channels that are symmetrical about the centerpoint (7 unique channels)
  //The tweeter values are in the LA_Tweeter_Delay_table_24
  //If the line array uses a different number of woofers or tweeters, you will need to include logic to select the proper table, but for now those tables are hard-coded

  const int number_of_woofers = 7;
  const int number_of_tweeters = 12;
  int table_row = 0;
  long table_offset = 0;
  uint8_t Shading_Data[4];

  for (int t = 0; t <= number_of_tweeters - 1; t++) {

    if (vol_taper > 4) {  //convex
      table_row = vol_taper - 4;
      table_offset = t * 4;
    }

    if (vol_taper <= 4) {
      table_row = 4 - vol_taper;
      table_offset = (number_of_tweeters - 1 - t) * 4;
    }

    Shading_Data[0] = pgm_read_byte_far(&(LA_Tweeter_Vol_table_24[table_row][table_offset]));
    Shading_Data[1] = pgm_read_byte_far(&(LA_Tweeter_Vol_table_24[table_row][table_offset + 1]));
    Shading_Data[2] = pgm_read_byte_far(&(LA_Tweeter_Vol_table_24[table_row][table_offset + 2]));
    Shading_Data[3] = pgm_read_byte_far(&(LA_Tweeter_Vol_table_24[table_row][table_offset + 3]));

    Lib_SPI.safeload_addr(tweeter_shading_addresses[t]);  //set up safeload address
    Lib_SPI.safeload_data(Shading_Data);                  //put the delay in the safeload registers
    Lib_SPI.safeload_xfer(1);                             //transfer the data
  }

  for (int w = 0; w <= number_of_woofers - 1; w++) {

    if (vol_taper > 4) {  //convex
      table_row = vol_taper - 4;
      table_offset = w * 4;
    }

    if (vol_taper <= 4) {
      table_row = 4 - vol_taper;
      table_offset = (number_of_woofers - 1 - w) * 4;
    }

    Shading_Data[0] = pgm_read_byte_far(&(LA_Woofer_Vol_table_14[table_row][table_offset]));
    Shading_Data[1] = pgm_read_byte_far(&(LA_Woofer_Vol_table_14[table_row][table_offset + 1]));
    Shading_Data[2] = pgm_read_byte_far(&(LA_Woofer_Vol_table_14[table_row][table_offset + 2]));
    Shading_Data[3] = pgm_read_byte_far(&(LA_Woofer_Vol_table_14[table_row][table_offset + 3]));

    Lib_SPI.safeload_addr(woofer_shading_addresses[w]);  //set up safeload address
    Lib_SPI.safeload_data(Shading_Data);                 //put the delay in the safeload registers
    Lib_SPI.safeload_xfer(1);                            //transfer the data
  }
}

void ADAU1466_cmd::Speaker_select(int speaker_code, word mute_addresses[3], word tweeter_addresses[12], word woofer_addresses[7]) {
  int mute_list[22] = { 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 };  //default is play all
  //0 = mute; 1 = play
  if (speaker_code > 3) {
    for (int i = 2; i < 22; i++) mute_list[i] = 0;  //if speaker code > 3, mute the sub and the individual drivers
  }

  switch (speaker_code) {
    case 0:
      break;  //if "play all" selected, go to end to update registers

    case 1:  //Tweeters
      mute_list[1] = 0;
      mute_list[2] = 0;  //mute woofers and sub
      break;
    case 2:  //Woofers
      mute_list[0] = 0;
      mute_list[2] = 0;  //mute tweeters and sub
      break;
    case 3:  //Sub
      mute_list[0] = 0;
      mute_list[1] = 0;  //mute tweeters and woofers
      break;

    case 4:
      mute_list[3] = 1;  //enable tweeter1
      break;
    case 5:
      mute_list[4] = 1;  //enable tweeter2
      break;
    case 6:
      mute_list[5] = 1;  //enable tweeter3
      break;
    case 7:
      mute_list[6] = 1;  //enable tweeter4
      break;
    case 8:
      mute_list[7] = 1;  //enable tweeter5
      break;
    case 9:
      mute_list[8] = 1;  //enable tweeter6
      break;
    case 10:
      mute_list[9] = 1;  //enable tweeter7
      break;
    case 11:
      mute_list[10] = 1;  //enable tweeter8
      break;
    case 12:
      mute_list[11] = 1;  //enable tweeter9
      break;
    case 13:
      mute_list[12] = 1;  //enable tweeter10
      break;
    case 14:
      mute_list[13] = 1;  //enable tweeter11
      break;
    case 15:
      mute_list[14] = 1;  //enable tweeter12
      break;
    case 16:
      mute_list[15] = 1;  //enable woofer1
      break;
    case 17:
      mute_list[16] = 1;  //enable woofer2
      break;
    case 18:
      mute_list[17] = 1;  //enable woofer3
      break;
    case 19:
      mute_list[18] = 1;  //enable woofer4
      break;
    case 20:
      mute_list[19] = 1;  //enable woofer5
      break;
    case 21:
      mute_list[20] = 1;  //enable woofer6
      break;
    case 22:
      mute_list[21] = 1;  //enable woofer7
      break;

    default:
      break;
  }
  //output_mute_list(mute_addresses[3], tweeter_addresses[12], woofer_addresses[7]);  //update mute registers

  uint8_t Cmd_Data[4] = { 0, 0, 0, 0 };
  int reg_number;

  for (int i = 0; i < 22; i++) {
    if (i < 3) reg_number = mute_addresses[i];
    if (i >= 3 && i < 15) reg_number = tweeter_addresses[i - 3];
    if (i >= 15) reg_number = woofer_addresses[i - 15];
    Cmd_Data[3] = mute_list[i];

    Lib_SPI.safeload_addr(reg_number);  //set up safeload address
    Lib_SPI.safeload_data(Cmd_Data);    //send volume data to safeload registers
    Lib_SPI.safeload_xfer(1);           //set the transfer bit
  }
}



/*Private routines*/

void ADAU1466_cmd::convert_Vol_to_8_24(const byte* Vol_table, byte v) {  //use current HCI value and volume look-up table to convert to ADAU1466 format
  long v4 = v + v + v + v;                                               //multiply by 4 to index the table

  Cmd_Data[0] = pgm_read_byte_far(Vol_table + v4);
  Cmd_Data[1] = pgm_read_byte_far(Vol_table + v4 + 1);
  Cmd_Data[2] = pgm_read_byte_far(Vol_table + v4 + 2);
  Cmd_Data[3] = pgm_read_byte_far(Vol_table + v4 + 3);
  Lib_SPI.safeload_data(Cmd_Data);  //send volume data to safeload registers
  Lib_SPI.safeload_xfer(1);         //set the transfer bit
}

void ADAU1466_cmd::float_to_safeload(int reg_num, float f) {
  //the float value is usually from 0 to 1.  The 8 overflow bits allow a greater range
  const float Two_to_the_24th = 16777216.0;
  uint32_t f_8_24 = f * Two_to_the_24th;
  uint8_t Cmd_Data[4];
  Cmd_Data[0] = lowByte((f_8_24 & 0xff000000) >> 24);
  Cmd_Data[1] = lowByte((f_8_24 & 0x00ff0000) >> 16);
  Cmd_Data[2] = lowByte((f_8_24 & 0x0000ff00) >> 8);
  Cmd_Data[3] = lowByte((f_8_24 & 0x000000ff));

  Lib_SPI.safeload_addr(reg_num);   //set up safeload address
  Lib_SPI.safeload_data(Cmd_Data);  //send volume data to safeload registers
  Lib_SPI.safeload_xfer(1);
  delay(1);  //wait for next audio sample
}

void ADAU1466_cmd::uint32_to_safeload(int reg_num, uint32_t i) {
  uint8_t Cmd_Data[4];
  Cmd_Data[0] = lowByte((i & 0xff000000) >> 24);
  Cmd_Data[1] = lowByte((i & 0x00ff0000) >> 16);
  Cmd_Data[2] = lowByte((i & 0x0000ff00) >> 8);
  Cmd_Data[3] = lowByte((i & 0x000000ff));

  Lib_SPI.safeload_addr(reg_num);   //set up safeload address
  Lib_SPI.safeload_data(Cmd_Data);  //send volume data to safeload registers
  Lib_SPI.safeload_xfer(1);
  delay(1);  //wait for next audio sample
}

void ADAU1466_cmd::Set_Vol_Trim_to_0(word reg_number) {
  uint8_t Cmd_Data[4];
  Lib_SPI.safeload_addr(reg_number);  //set up safeload address
  Cmd_Data[0] = 0;
  Cmd_Data[1] = 0;
  Cmd_Data[2] = 0;
  Cmd_Data[3] = 0;
  Lib_SPI.safeload_data(Cmd_Data);  //send volume data to safeload registers
  Lib_SPI.safeload_xfer(1);         //set the transfer bit
}

void ADAU1466_cmd::convert_delay_level(byte delay_level) {
  uint8_t Delay_Data[4];
  Delay_Data[0] = 0;
  Delay_Data[1] = 0;
  Delay_Data[2] = 0;
  Delay_Data[3] = pgm_read_byte_far(Delay_table + delay_level);
  Lib_SPI.safeload_data(Delay_Data);  //put the delay in the safeload registers
  Lib_SPI.safeload_xfer(1);           //transfer the data
}

void ADAU1466_cmd::Set_invert_to_on(word reg_number) {
  uint8_t Cmd_Data[4];
  Mute();
  Lib_SPI.safeload_addr(reg_number);  //set up safeload address
  Cmd_Data[0] = 0;
  Cmd_Data[1] = 0;
  Cmd_Data[2] = 0;
  Cmd_Data[3] = 1;
  Lib_SPI.safeload_data(Cmd_Data);  //send data to safeload registers
  Lib_SPI.safeload_xfer(1);         //set the transfer bit

  Lib_SPI.safeload_addr(reg_number + 1);  //set up the second safeload address
  Lib_SPI.safeload_data(Cmd_Data);        //the same data is sent to both addresses
  Lib_SPI.safeload_xfer(1);               //set the transfer bit
  delay(delay_Mid);
  Unmute();
}

void ADAU1466_cmd::Set_invert_to_off(word reg_number) {
  uint8_t Cmd_Data[4];
  Mute();
  Lib_SPI.safeload_addr(reg_number);  //set up safeload address
  Cmd_Data[0] = 0;
  Cmd_Data[1] = 0;
  Cmd_Data[2] = 0;
  Cmd_Data[3] = 0;
  Lib_SPI.safeload_data(Cmd_Data);  //send data to safeload registers
  Lib_SPI.safeload_xfer(1);         //set the transfer bit

  Lib_SPI.safeload_addr(reg_number + 1);  //set up the second safeload address
  Lib_SPI.safeload_data(Cmd_Data);        //the same data is sent to both addresses
  Lib_SPI.safeload_xfer(1);               //set the transfer bit
  delay(delay_Mid);
  Unmute();
}

void ADAU1466_cmd::Mute_all(word _Mute_addresses[number_of_mute_addresses], int _mute_state) {


  if (_mute_state == 0) {
    //if (_mute_state == 0 && setup_done == true) {
    for (int i = 0; i < number_of_mute_addresses; i++) {
      Set_Mute_to_1(_Mute_addresses[i]);
    }
    //Serial.println("unmuted");

  } else {
    for (int i = 0; i < number_of_mute_addresses; i++) {
      Set_Mute_to_0(_Mute_addresses[i]);
    }
    //Serial.println("muted");
  }
}


void ADAU1466_cmd::Mute() {
  Mute_all(_copy_of_Mute_addresses, 1);
}

void ADAU1466_cmd::Unmute() {
  if(setup_done) Mute_all(_copy_of_Mute_addresses, 0);
}

void ADAU1466_cmd::Set_Mute_to_0(word reg_number) {  //this will mute the audio
  uint8_t Cmd_Data[4];
  Lib_SPI.safeload_addr(reg_number);  //set up safeload address

  Cmd_Data[0] = 0;
  Cmd_Data[1] = 0;
  Cmd_Data[2] = 0;
  Cmd_Data[3] = 0;
  Lib_SPI.safeload_data(Cmd_Data);  //send volume data to safeload registers
  Lib_SPI.safeload_xfer(1);         //set the transfer bit
}


void ADAU1466_cmd::Set_Mute_to_1(word reg_number) {  //this will unmute the audio
  uint8_t Cmd_Data[4];
  Lib_SPI.safeload_addr(reg_number);  //set up safeload address

  Cmd_Data[0] = 0;
  Cmd_Data[1] = 0;
  Cmd_Data[2] = 0;
  Cmd_Data[3] = 1;
  Lib_SPI.safeload_data(Cmd_Data);  //send volume data to safeload registers
  Lib_SPI.safeload_xfer(1);         //set the transfer bit
}


float ADAU1466_cmd::Convert_BSC_freq_code(int freq_code) {
  return BSC_Freq_table[freq_code];
}

float ADAU1466_cmd::Convert_BSC_gain_code(int gain_code) {
  return BSC_Gain_table[gain_code];
}

float ADAU1466_cmd::Convert_EQ_gain_code(int gain_code) {
  return EQ_Gain_table[gain_code];
}

void ADAU1466_cmd::update_driver_delay(word driver_delay_address, byte delay_data) {
  //Show us what you got
  //Serial.print("Address:  ");
  //Serial.print(driver_delay_address, HEX);
  //Serial.print("   Data:  ");
  //Serial.println(delay_data, HEX);

  uint8_t Delay_Data[4];
  Delay_Data[0] = 0;
  Delay_Data[1] = 0;
  Delay_Data[2] = 0;
  Delay_Data[3] = delay_data;

  Lib_SPI.safeload_addr(driver_delay_address);  //set up safeload address
  Lib_SPI.safeload_data(Delay_Data);            //put the delay in the safeload registers
  Lib_SPI.safeload_xfer(1);                     //transfer the data
}


/*debug stuff to copy and paste...
  for (int i = 0; i < 10; i++) {

  Serial.println(i);
  Serial.print("  number_of_values:  ");
  Serial.print(freq_code);
  Serial.print("  ");
  Serial.println(freq_code);

  }
*/