/*
  This code controls the ADAU1466 in the 20 channel line array amp.

  It receives commands from a Bluetooth BLE interface, executes the commands, and
  then echos the command and the value to be displayed on the Bluetooth client.
  It also echos the raw commands out to the SPDIF user bits.  This code is for the 
  left speaker in the stereo pair, as there is delay required to make sure the channels
  are synchronized. 

  This code also controls and monitors the SSM3582 amplifier chips.  Any errors are
  reported out the serial port.  The ADAU1467 version of this code also uses the
  OLED display to report any errors.  The ADAU1466 amp uses TDM8, whereas the ADAU1467
  version used TDM4.  Therefore, the SSM3582 programming is different between these versions.

  This code is to be used with this SigmaStudio design:  ADAU1466_line_array3.dspproj
  This code is to be used with this Android App:  Line_array

  This main file includes the global variables and definitions, the startup code,
  and the main loop.
*/

/*-----------------------------------------------------------------------------------------
   External libraries
  ----------------------------------------------------------------------------------------*/

#include <Wire.h>
#include <SPI.h>
#include <Preferences.h>
#include <BLEDevice.h>
#include <BLEUtils.h>
#include <BLEServer.h>
#include <BLE2902.h>



char SECRET_SSID[] = "Old_side";
char SECRET_OPTIONAL_PASS[] = "7034357359";
char SECRET_DEVICE_KEY[] = "oldside";
//#include "thingProperties.h"


/*-----------------------------------------------------------------------------------------
   ADAU1466_Audiodevelopers libraries (these must be in the libraries folder) and
   other files that must be in the project folder
  ----------------------------------------------------------------------------------------*/
#include "ADAU1466_cmd.h"  //library routines to "execute" the DSP functions
#include <ADAU1466_DSP.h>  //library routines for DSP functions
#include <ADAU1466_SPI.h>  //library routines for DSP functions
#include <ADAU1466_Shared_enums.h>
#include "Cell_map.h"     //This is the cell definitions generated by a .NET program that processes the Sigma Studio PARAM.h file
#include "enum_struct.h"  //see this file for the program-specific enumerations and structures

/*-----------------------------------------------------------------------------------------
   Pin Assignments
  ----------------------------------------------------------------------------------------*/
//const int led = 2;
//const int A28_reset = 1;
//const int ADAU1466_reset = 3;

#define SDA_pin 6
#define SCL_pin 7
#define SPI_SCK 8
#define SPI_MISO 9
#define SPI_MOSI 10
#define SPI_SS 20
#define ADAU1466_reset 4
#define SPDIF_sel 5

/*-----------------------------------------------------------------------------------------
   I2C/SPI Addresses and devices
  ----------------------------------------------------------------------------------------*/
//const uint8_t ADAU1466_addr = 0x3B;   //ADDR0 and ADDR1 are INPUT_PULLUP, so address is 38 + 3 = 3B
const uint8_t ADAU1466_addr = 0x3B;
#define SSM3582_I2C_ADDR0 0x10
#define SSM3582_I2C_ADDR1 0x11
#define EEPROM_ADDR 0x50
#define number_of_SSM3582_chips 10
uint8_t SSM3582_addresses[16] = { 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F };
long SSM3582_timer = millis();
bool SSM3582_powered_on = true;
bool SSM3582_previous_state = SSM3582_powered_on;

//ADAU1466_I2C Lib_I2C(ADAU1466_addr, SafeLoadModule);  //start an instance of the I2C library
ADAU1466_SPI Lib_SPI = ADAU1466_SPI(SafeLoadModule, SPI_SCK, SPI_MISO, SPI_MOSI, SPI_SS, &SPI);  //start an instance of the SPI library

/*-----------------------------------------------------------------------------------------
   Bluetooth BLE UUID's
  ----------------------------------------------------------------------------------------*/
#define SERVICE_UUID "6E400001-B5A3-F393-E0A9-E50E24DCCA9E"  // UART service UUID
#define CHARACTERISTIC_UUID_RX "6E400002-B5A3-F393-E0A9-E50E24DCCA9E"
#define CHARACTERISTIC_UUID_TX "6E400003-B5A3-F393-E0A9-E50E24DCCA9E"


/*-----------------------------------------------------------------------------------------
   Global Variables for HCI and SCI Routines
  ----------------------------------------------------------------------------------------*/
char Command;  //A single character indicating the command type
uint8_t Parameter_1;
uint8_t Parameter_1_MSB = 0x08;
uint8_t Parameter_2;
byte Command_length = 0;
byte Cmd_count;
const byte number_of_commands = 16;
commands CMD[number_of_commands];
const uint8_t cmd_buffer_length = 16;
String message;  //string that stores the incoming message
byte current_field_ID;
bool setup_done = false;
char field_string_global[cmd_buffer_length + 1];
struct field HCI_state[Last_field_ID];
struct commands cmd[26];
struct bass_enhancement_options BE;  //make an instance of the bass enhancement structure called BE

bool External_EEPROM = false;      //make this "false" for boards that don't have an external EEPROM
bool Invert_back_channels = true;  //determines the setting of the inverter in the Dipole module

/*-----------------------------------------------------------------------------------------
   Bluetooth BLE variables
  ----------------------------------------------------------------------------------------*/
BLECharacteristic* pCharacteristic;
bool BLE_enabled = true;
bool BLEdeviceConnected = false;
//std::string rxValue = "";
String rxValue = "";
bool rx_flag = false;
long rx_timer = 0;
long tx_timer = 0;
#define RX_revisit_time 10
#define TX_revisit_time 50
int SPDIF_RX_state = 0;

/*-----------------------------------------------------------------------------------------
   BLE Callbacks
  ----------------------------------------------------------------------------------------*/
class MyServerCallbacks : public BLEServerCallbacks {
  void onConnect(BLEServer* pServer) {
    BLEdeviceConnected = true;
    Serial.println("connected");
  };
  void onDisconnect(BLEServer* pServer) {
    BLEdeviceConnected = false;
    Serial.println("disconnected");

    //Make sure the server is still visible
    pServer->getAdvertising()->start();
  }
};

class MyCallbacks : public BLECharacteristicCallbacks {
  void onWrite(BLECharacteristic* pCharacteristic) {
    rxValue = pCharacteristic->getValue();
    rx_flag = true;
    for (int i = 0; i < rxValue.length(); i++) {
      Serial.print(rxValue[i]);
    }
    Serial.println(" ");
  }
};

//Instantiate the Preferences object
Preferences preferences;

/*-----------------------------------------------------------------------------------------
   Initialization -- setup()
  ----------------------------------------------------------------------------------------*/

void setup() {
  delay(1000);  //let devices boot up

  Port_init();  //set up port pins and apply reset to the ADAU1466

  SPI_port_init();
  Lib_SPI.begin();

  setup_done = false;  //this flag used to mute the ADC/DAC's during initialization


  Wire.begin(SDA_pin, SCL_pin, 400000);  // join I2C bus as the master to control the SSM3582 chips

  Serial.begin(115200);  //needed for serial debugger

  BLE_Init();  //see CPU_init.ino

  // Defined in thingProperties.h
  // initProperties();

  // Connect to Arduino IoT Cloud
  //ArduinoCloud.begin(ArduinoIoTPreferredConnection);

  //setDebugMessageLevel(2);
  //ArduinoCloud.printDebugInfo();


  //ADAU1466_I2C.ADAU1466_soft_reset();

  HCI_init();  //set up the HCI table and initialize the display
  CMD_init();  //load the CMD data

  initialization_mute();

  //save_state();           //uncomment this to put default values into the EEPROM

  restore_state();  //load the saved values from Preferences

  setup_done = true;

  init_SSM3582(SSM3582_addresses);

  initialization_unmute();

  Serial.println("state restored");

  check_SSM3582(SSM3582_addresses);  //debug:  check to make sure all chips respond

}

/*-----------------------------------------------------------------------------------------
   The main loop
  ----------------------------------------------------------------------------------------*/

void loop() {
  // main loop:  check for serial input, key switch entry, dipswitch changes, and elapsed timers

  //ArduinoCloud.update();

  SCI_check();  //see SCI.ino--not used with BLE

  if (rx_flag == true) {
    Serial.println("Serial input detected  ");
    BLE_RX_check();
  }


  if ((tx_timer != 0) && (millis() > tx_timer)) {  //if timer enabled (non-zero) and current time greater than 10 milliseconds after setting tx_timer
    SPDIF_TX_clear();                              //clear out the command
    tx_timer = 0;                                  //disable timer until another command comes in
  }

  if (millis() > rx_timer) {
    SPDIF_check();  //see if SPDIF RX state needs to change.  This routine is in SCI_BLE.ino
  }

  if (millis() > SSM3582_timer) {
    for (int i = 0; i < number_of_SSM3582_chips; i++) {
      word SSM3582_status = check_SSM3582_status(SSM3582_addresses[i]);

      //SSM3582_status = 0;

      if (SSM3582_status != 0) {  //only print status if errors show up
        Serial.print("I2C address = ");
        Serial.print(SSM3582_addresses[i], HEX);
        Serial.print("  Status = ");
        Serial.println(SSM3582_status, HEX);
      }
    }
    //Serial.println("writing to OLED");
    //display.clear();
    //display.setCursor(0, 0);
    //display.print("Hello");
    //display.display();
    SSM3582_timer = millis() + 1000;  //check once every second
  }
  //check_SSM3582_power(SSM3582_addresses[0]);                          //debug
  SSM3582_powered_on = check_SSM3582_power_on(SSM3582_addresses[0]);  //read power-on status of amp #1
  if (SSM3582_powered_on != SSM3582_previous_state) {
    if (SSM3582_powered_on == true) Serial.println("Amp1 is now powered on");
    if (SSM3582_powered_on == false) Serial.println("Amp1 is now turned off");
    SSM3582_previous_state = SSM3582_powered_on;
  }
}
