/*
  ADAU1466_DSP.cpp - Library of DSP functions for the ADAU1466
  Created by Neil Davis, December, 2020.
*/

#include "Arduino.h"
#include "ADAU1466_DSP.h"
#include "ADAU1466_Lookup_tables.h"
#include <ADAU1466_SPI.h>
#include <ADAU1466_Bilinear.h>
#include <ADAU1466_Shared_enums.h>


const uint8_t ADAU1466_addr = 0x3B;
const word SafeLoadModule = 24576;

const byte max_BQ5_biquads = 5;     //maximum number of biquads used.  This constant is needed to "clear out" unused biquads
const byte max_BQ3_biquads = 3;
float _ADAU1466_coef_array[6];         //array to hold one set of biquad coefficents processed by Bilinear() and sent to coef_to_ADAU1466()

extern ADAU1466_SPI Lib_SPI;

ADAU1466_Bilinear My_ADAU1466_Bilinear;

ADAU1466_DSP::ADAU1466_DSP()  {}

/* Public Functions */

float ADAU1466_DSP::convert_freq_code(Shared_enum::Xover_drivers driver_set, byte freq_code) {
  switch (driver_set) {
    case Shared_enum::ST_T:
      return ST_T_Xover_freq_table[freq_code];    //look up the Super Tweeter-Tweeter frequency in the table
    case Shared_enum::T_M:
      return T_M_Xover_freq_table[freq_code];    //look up the Tweeter-Midrange frequency in the table
    case Shared_enum::M_W:
      return M_W_Xover_freq_table[freq_code];    //look up the Midrange-Woofer frequency in the table
    case Shared_enum::W_S:
      return W_S_Xover_freq_table[freq_code];    //look up the Woofer-Sub frequency in the table
    case Shared_enum::T_W:  
      return T_W_Xover_freq_table[freq_code];    //look up the Woofer-Sub frequency in the table
    default:
      return W_S_Xover_freq_table[1]; 
  }
}

Shared_enum::Xovers ADAU1466_DSP::convert_Xover_code(Shared_enum::Xover_drivers driver_set, byte type_code) {
  Shared_enum::Xovers x_table[6] = {Shared_enum::none, Shared_enum::BW_1, Shared_enum::BW_3, Shared_enum::LR_2, Shared_enum::LR_4, Shared_enum::LR_8};
  x_type = x_table[type_code];     //there is a one-to-one mapping of code to enumerated type so no look-up is needed
/*
          Serial.print(" type_code =  ");
          Serial.print(type_code);
          Serial.print("  ");
          Serial.print(x_type);
          Serial.print("  ");
*/

  return x_type;
}

void ADAU1466_DSP::BQ5_Xover_calc(Shared_enum::BQ_spec BQ, word BQ5_addresses[5][2]) {
  /* This is where the crossover is calculated.  The input is a biquad_spec object, which describes the crossover filter.
      The biquad_spec has the following information:
        --filter frequency
        --the number of biquads needed to implement the crossover
        --the filter order of each biquad
        --the Q of each biquad
      The crossover_calc routine calls the biquad_load routine to calculate the biquad coefficients and puts the coefficient data in Parameter RAM
  */

  for (byte n = 0; n < BQ.biquads_needed; n++) {
    BQ5_biquad_load(n, BQ.Freq_center, BQ.filter_order[n], BQ.filter_q[n], BQ5_addresses);
  }
  fill_BQ5_with_none(BQ.biquads_needed, BQ5_addresses);                      //"zero-out" the remaining biquads
}



void ADAU1466_DSP::BQ5_biquad_load(byte biquad, float Freq, byte order, float q, word BQ5_addresses[5][2])
/*
   This routine calculates the coefficients needed to implement the desired filter by calling the coef_calc routine.
   It then converts the coefficients from floating point to 8.24 integer in the coef_to_ADAU1466 function.  It then
   loads the data into the ADAU1466 Parameter RAM using the safe load registers, so that all of the data is updated
   during the same audio sample.  The two lowpass filters are written, followed by the highpass filters
*/
{
  Shared_enum::tfilters filtertype;
  if (order == 1) filtertype = Shared_enum::One_Pole_LP; else filtertype = Shared_enum::Two_Pole_LP;


  Serial.print("LP Filter_enum = ");
  Serial.print(filtertype);
  Serial.print("   Order = ");
  Serial.print(order);
  Serial.print("  BQ_addr = ");
  Serial.println(BQ5_addresses[biquad][0]);



  My_ADAU1466_Bilinear.coef_calc(Freq, 1.0, q, filtertype, _ADAU1466_coef_array);
  coef_to_ADAU1466(_ADAU1466_coef_array);
  Lib_SPI.biquad_safe_load(BQ5_addresses[biquad][0], ADAU1466_coefs);     //TW, lowpass

  if (order == 1) filtertype = Shared_enum::One_Pole_HP; else filtertype = Shared_enum::Two_Pole_HP;
  //Q already set


  Serial.print("HP Filter_enum = ");
  Serial.print(filtertype);
  Serial.print("   Order = ");
  Serial.print(order);
  Serial.print("  BQ_addr = ");
  Serial.println(BQ5_addresses[biquad][1]);



  My_ADAU1466_Bilinear.coef_calc(Freq, 1.0, q, filtertype, _ADAU1466_coef_array);
  coef_to_ADAU1466(_ADAU1466_coef_array);

  Lib_SPI.biquad_safe_load(BQ5_addresses[biquad][1], ADAU1466_coefs);     //TW, highpass

}

void ADAU1466_DSP::BQ3_Xover_calc(Shared_enum::BQ_spec BQ, word BQ3_addresses[3][2]) {
  /* This is where the crossover is calculated.  The input is a biquad_spec object, which describes the crossover filter.
      The biquad_spec has the following information:
        --filter frequency
        --the number of biquads needed to implement the crossover
        --the filter order of each biquad
        --the Q of each biquad
      The crossover_calc routine calls the biquad_load routine to calculate the biquad coefficients and puts the coefficient data in Parameter RAM
  */

  for (byte n = 0; n < BQ.biquads_needed; n++) {
    BQ3_biquad_load(n, BQ.Freq_center, BQ.filter_order[n], BQ.filter_q[n], BQ3_addresses);
  }
  fill_BQ3_with_none(BQ.biquads_needed, BQ3_addresses);                      //"zero-out" the remaining biquads
}

void ADAU1466_DSP::BQ3_biquad_load(byte biquad, float Freq, byte order, float q, word BQ3_addresses[3][2])
/*
   This routine calculates the coefficients needed to implement the desired filter by calling the coef_calc routine.
   It then converts the coefficients from floating point to 28-bit integer in the coef_to_ADAU1466 function.  It then
   loads the data into the ADAU1466 Parameter RAM using the safe load registers, so that all of the data is updated
   during the same audio sample.  The two lowpass filters are written, followed by the highpass filters
*/
{
  Shared_enum::tfilters filtertype;
  if (order == 1) filtertype = Shared_enum::One_Pole_LP; else filtertype = Shared_enum::Two_Pole_LP;
  My_ADAU1466_Bilinear.coef_calc(Freq, 1.0, q, filtertype, _ADAU1466_coef_array);
  coef_to_ADAU1466(_ADAU1466_coef_array);

  Lib_SPI.biquad_safe_load(BQ3_addresses[biquad][0], ADAU1466_coefs);     //WS, Left channel, lowpass
  //Lib_SPI.biquad_safe_load(WS_addresses[biquad][1], ADAU1466_coefs);     //WS, Right channel, biquad #1, lowpass


/*
      Serial.print(BQ3_addresses[biquad][0]);
      Serial.print("LP:  ");
      for (byte i = 0; i < 5; i++) {      //5 coefficients
        for (int j = 0; j < 4; j++) {     //byte counter for 32-bit data
          Serial.print(ADAU1466_coefs[i][j]);
          Serial.print(" ");
        }
        Serial.print("   ");
      }
      Serial.println(" ");
*/


  if (order == 1) filtertype = Shared_enum::One_Pole_HP; else filtertype = Shared_enum::Two_Pole_HP;
  //Q already set
  My_ADAU1466_Bilinear.coef_calc(Freq, 1.0, q, filtertype, _ADAU1466_coef_array);
  coef_to_ADAU1466(_ADAU1466_coef_array);

  Lib_SPI.biquad_safe_load(BQ3_addresses[biquad][1], ADAU1466_coefs);     //WS, Left channel, highpass
  //Lib_SPI.biquad_safe_load(WS_addresses[biquad][3], ADAU1466_coefs);     //WS, Right channel, highpass


  /*    
      Serial.print(BQ3_addresses[biquad][1]);
      Serial.print(":  ");
      for (byte i = 0; i < 5; i++) {      //5 coefficients
        for (int j = 0; j < 4; j++) {     //byte counter for 32-bit data
          Serial.print(ADAU1466_coefs[i][j]);
          Serial.print(" ");
        }
        Serial.print("   ");
      }
      Serial.println(" ");
*/



}

void ADAU1466_DSP::filter_calc(Shared_enum::Filter_spec Filter, word BQ_address) {

  My_ADAU1466_Bilinear.coef_calc(Filter.Freq_center, Filter.Gain, Filter.Q, Filter.filter_type, _ADAU1466_coef_array);
  coef_to_ADAU1466(_ADAU1466_coef_array);
  Lib_SPI.biquad_safe_load(BQ_address, ADAU1466_coefs);
}


Shared_enum::BQ_spec ADAU1466_DSP::get_biquad_spec(float FC, Shared_enum::Xovers x_type)
/*
   This routine does a "look-up" on the enumerated crossover (X_type) to find the filter order and Q.  This "look-up" is done
   using a long case switch-case statement.  The current HCI only allows 6 different crossover types, but the switch-case statement
   actually supports 18 different crossover types, as shown below.  These can be easily added to the HCI table.  The biquad_spec
   structure is already specified to allow using 7 biquads, so any of the crossover types listed below could be implemented by just
   changing the HCI
*/
{
  struct Shared_enum::BQ_spec biquads;         //make an instance of the biquad structure called "biquads"
  biquads.Freq_center = FC;

  switch (x_type) {
    case none:
      biquads.biquads_needed = 0;
      break;

    case BW_1:
      biquads.biquads_needed = 1;
      biquads.filter_order[0] = 1;
      biquads.filter_q[0] = 1.0;
      break;

    case BW_2:
      biquads.biquads_needed = 1;
      biquads.filter_order[0] = 2;
      biquads.filter_q[0] = 0.707;
      break;

    case BW_3:
      biquads.biquads_needed = 2;
      biquads.filter_order[0] = 1;    biquads.filter_order[1] = 2;
      biquads.filter_q[0] = 0.5;      biquads.filter_q[1] = 1.0;
      break;

    case BW_4:
      biquads.biquads_needed = 2;
      biquads.filter_order[0] = 2;    biquads.filter_order[1] = 2;
      biquads.filter_q[0] = 0.54;     biquads.filter_q[1] = 1.31;
      break;

    case BW_5:
      biquads.biquads_needed = 3;
      biquads.filter_order[0] = 1;    biquads.filter_order[1] = 2;    biquads.filter_order[2] = 2;
      biquads.filter_q[0] = 0.5;      biquads.filter_q[1] = 0.62;     biquads.filter_q[2] = 1.62;
      break;

    case BW_6:
      biquads.biquads_needed = 3;
      biquads.filter_order[0] = 2;    biquads.filter_order[1] = 2;    biquads.filter_order[2] = 2;
      biquads.filter_q[0] = 0.52;     biquads.filter_q[1] = 0.71;     biquads.filter_q[2] = 1.93;
      break;

    case Bes_2:
      biquads.biquads_needed = 1;
      biquads.filter_order[0] = 2;    biquads.filter_q[0] = 0.58;
      break;

    case Bes_4:
      biquads.biquads_needed = 2;
      biquads.filter_order[0] = 2;    biquads.filter_order[1] = 2;
      biquads.filter_q[0] = 0.52;     biquads.filter_q[1] = 0.81;
      break;

    case Bes_5:
      biquads.biquads_needed = 3;
      biquads.filter_order[0] = 1;    biquads.filter_order[1] = 2;    biquads.filter_order[2] = 2;
      biquads.filter_q[0] = 0.5;      biquads.filter_q[1] = 0.56;     biquads.filter_q[2] = 0.92;
      break;

    case Bes_6:
      biquads.biquads_needed = 3;
      biquads.filter_order[0] = 2;    biquads.filter_order[1] = 2;    biquads.filter_order[2] = 2;
      biquads.filter_q[0] = 0.51;     biquads.filter_q[1] = 0.61;     biquads.filter_q[2] = 1.02;
      break;

    case LR_2:
      biquads.biquads_needed = 1;
      biquads.filter_order[0] = 2;
      biquads.filter_q[0] = 0.5;
      break;

    case LR_4:
      biquads.biquads_needed = 2;
      biquads.filter_order[0] = 2;    biquads.filter_order[1] = 2;
      biquads.filter_q[0] = 0.707;    biquads.filter_q[1] = 0.71;
      break;

    case LR_6:
      biquads.biquads_needed = 3;
      biquads.filter_order[0] = 2;    biquads.filter_order[1] = 2;    biquads.filter_order[2] = 2;
      biquads.filter_q[0] = 0.5;      biquads.filter_q[1] = 1.0;      biquads.filter_q[2] = 1.0;
      break;

    case LR_8:
      biquads.biquads_needed = 4;
      biquads.filter_order[0] = 2;    biquads.filter_order[1] = 2;    biquads.filter_order[2] = 2;    biquads.filter_order[3] = 2;
      biquads.filter_q[0] = 0.54;     biquads.filter_q[1] = 0.54;     biquads.filter_q[2] = 1.31;     biquads.filter_q[3] = 1.31;
      break;

    case LR_10:
      biquads.biquads_needed = 5;
      biquads.filter_order[0] = 2;  biquads.filter_order[1] = 2;  biquads.filter_order[2] = 2;
      biquads.filter_order[3] = 2;  biquads.filter_order[4] = 2;

      biquads.filter_q[0] = 0.5;    biquads.filter_q[1] = 0.62;   biquads.filter_q[2] = 0.62;
      biquads.filter_q[3] = 1.62;   biquads.filter_q[4] = 1.62;
      break;
  }
  return  biquads;
}


/*  Private Functions  */

void ADAU1466_DSP::coef_to_ADAU1466(float coef_array[6]) {
  //convert scaled floating point filter coefficients to bytes that can be loaded into the ADAU1466 Parameter RAM
  const float Two_to_the_24th = 16777216.0;
  const long bits_32 = 0xffffffff;
  long coef_long;

  for (byte i = 0; i < 5; i++) {
//Serial.println(coef_array[i], 8);
    coef_array[i] = coef_array[i] * Two_to_the_24th;
    coef_long = long(coef_array[i]);
    coef_long = coef_long & bits_32;
    ADAU1466_coefs [i][0] = lowByte((coef_long & 0xff000000) >> 24);
    ADAU1466_coefs [i][1] = lowByte((coef_long & 0x00ff0000) >> 16);
    ADAU1466_coefs [i][2] = lowByte((coef_long & 0x0000ff00) >> 8);
    ADAU1466_coefs [i][3] = lowByte((coef_long & 0x000000ff));
  }
}


void ADAU1466_DSP::fill_BQ5_with_none(byte number_of_biquads, word BQ5_addresses[5][2])
/*      This routine simply fills in the unused biquads with "1"--that is, the filter is a pass-through.
     The coef_passthrough routine sets the coefficients to be "1", and the switch-case statement is executed for
     each biquad, until the maximum number is reached (7).
*/

{
  for (byte n = number_of_biquads; n < max_BQ5_biquads; n++)
  {
    //Serial.print(" TW fill with none biquad_counter = ");
    //Serial.println(n);
    coef_passthrough();   //Put a "1" (passthrough) in the coef array
    for (int j = 0; j < 2; j++) {
      Lib_SPI.biquad_safe_load(BQ5_addresses[n][j], BQ_coefs);
      
/*
      Serial.print("number_of_biquads = ");
      Serial.println(number_of_biquads);
      Serial.print(BQ5_addresses[n][j]);
      Serial.print(" ");
      for (byte i = 0; i < 5; i++) {  
        for (int j = 0; j < 4; j++) {
          Serial.print(BQ_coefs[i][j]);
          Serial.print(" ");
        }
        Serial.print("   ");
      }
      Serial.println(" ");
*/

    }
  }
}

void ADAU1466_DSP::fill_BQ3_with_none(byte number_of_biquads, word BQ3_addresses[3][2]) {
  /*   This routine simply fills in the unused biquads with "1"--that is, the filter is a pass-through.
       The coef_passthrough routine sets the coefficients to be "1", and the switch-case statement is executed for
       each biquad, until the maximum number is reached (7).
  */

  for (byte n = number_of_biquads; n < max_BQ3_biquads; n++)
  {
    coef_passthrough();
    for (int j = 0; j < 2; j++) {
      Lib_SPI.biquad_safe_load(BQ3_addresses[n][j], BQ_coefs);

/*
      Serial.print("number_of_biquads = ");
      Serial.println(number_of_biquads);
      Serial.print(BQ3_addresses[n][j]);
      Serial.print(" ");
      for (byte i = 0; i < 5; i++) {      //5 coefficients
        for (int j = 0; j < 4; j++) {     //byte counter for 32-bit data
          Serial.print(BQ_coefs[i][j]);
          Serial.print(" ");
        }
        Serial.print("   ");
      }
      Serial.println(" ");
*/

    }
  }
}


void ADAU1466_DSP::coef_passthrough() {        //Put a "1" in the biquad
  for (byte i = 0; i < 5; i++) {        //i is the coefficient counter for the biquad
    for (byte j = 0; j < 4; j++ ) {     //J is the byte counter for the 32-bit data
      BQ_coefs [i][j] = 0x00;
    }
    //Coefficient B0 is in the third position in the ADAU1466
    BQ_coefs [2][0] = 0x01;
  }

}

