#include <ADAU1466_Bilinear.h>

ADAU1466_Bilinear::ADAU1466_Bilinear() {}

void ADAU1466_Bilinear::coef_calc(float FC, float gain, float Q, Shared_enum::tfilters filtertype, float (&coef_array)[6])
{
  // Entry requirements:
  //     filtertype, FC, Q, gain and Fs must be set
  // intermediate values needed for the coefficient calculations

  w = 2.0 * PI * FC / Fs;
  wS = sin(w);
  wC = cos(w);
  alpha = wS / (2.0 * Q);
  A = sqrt(pow(10.0, (gain / 20.0)));
  beta = sqrt(A) / Q;

  // Coefficient order is different from ADAU1701:  B2, B1, B0, A2, A1 vs b0, b1, b2, a1, a2

  coef_array[2] = coefb0(filtertype);
  coef_array[1] = coefb1(filtertype);
  coef_array[0] = coefb2(filtertype);
  coef_array[3] = coefa0(filtertype);
  coef_array[5] = coefa1(filtertype);
  coef_array[4] = coefa2(filtertype);

  // scale the array so that a0 = 1
  scalefactor = coef_array[3];
  if (scalefactor == 0)
    scalefactor = 1;
  for (byte i = 0; i < 6; i++)
  {
    coef_array[i] = coef_array[i] / scalefactor;
  }
  // skip over A0 and make the A coefficients negative
  coef_array[3] = -coef_array[4];
  coef_array[4] = -coef_array[5];

  /*
  Serial.print("B0 = ");
  Serial.println(coef_array[2],4);
  Serial.print("B1 = ");
  Serial.println(coef_array[1],4);
  Serial.print("B2 = ");
  Serial.println(coef_array[0],4);
  Serial.print("A1 = ");
  Serial.println(coef_array[4],4);
  Serial.print("A2 = ");
  Serial.println(coef_array[3],4);
  */
}

float ADAU1466_Bilinear::coefb0(Shared_enum::tfilters filtertype)
{
  switch (filtertype)
  {

  case Shared_enum::One_Pole_LP:
    k = tan(w / 2.0);
    alpha = 1.0 + k;
    return (k / alpha);

  case Shared_enum::One_Pole_HP:
    k = tan(w / 2.0);
    alpha = 1.0 + k;
    return (1.0 / alpha);

  case Shared_enum::Two_Pole_LP:
    return ((1.0 - wC) / 2.0);

  case Shared_enum::Two_Pole_HP:
    return ((1.0 + wC) / 2.0);

  case Shared_enum::LP_Shelf:
    return (A * ((A + 1.0) - ((A - 1.0) * wC) + (beta * wS)));

  case Shared_enum::HP_Shelf:
    return (A * ((A + 1.0) + ((A - 1.0) * wC) + (beta * wS)));

  case Shared_enum::Bandpass:
    return alpha;

  case Shared_enum::Notch:
    return 1.0;

  case Shared_enum::Peaking:
    return (1.0 + (alpha * A));

  case Shared_enum::All_Pass:
    return (1.0 - alpha);

  case Shared_enum::None:
    return 1.0;

  default:
    return 1.0;
  }
}

float ADAU1466_Bilinear::coefb1(Shared_enum::tfilters filtertype)
{
  switch (filtertype)
  {
  case Shared_enum::One_Pole_LP:
    k = tan(w / 2.0);
    alpha = 1.0 + k;
    return (k / alpha);

  case Shared_enum::One_Pole_HP:
    k = tan(w / 2.0);
    alpha = 1.0 + k;
    return (-1.0 / alpha);

  case Shared_enum::Two_Pole_LP:
    return (1.0 - wC);

  case Shared_enum::Two_Pole_HP:
    return -(1.0 + wC);

  case Shared_enum::LP_Shelf:
    return (2.0 * A * ((A - 1.0) - ((A + 1.0) * wC)));

  case Shared_enum::HP_Shelf:
    return (-2.0 * A * ((A - 1.0) + ((A + 1.0) * wC)));

  case Shared_enum::Bandpass:
    return 0.0;

  case Shared_enum::Notch:
    return (-2.0 * wC);

  case Shared_enum::Peaking:
    return (-2.0 * wC);

  case Shared_enum::All_Pass:
    return (-2.0 * wC);

  case Shared_enum::None:
    return 0.0;

  default:
    return 0.0;
  }
}

float ADAU1466_Bilinear::coefb2(Shared_enum::tfilters filtertype)
{
  switch (filtertype)
  {
  case Shared_enum::One_Pole_LP:
    return 0.0;

  case Shared_enum::One_Pole_HP:
    return 0.0;

  case Shared_enum::Two_Pole_LP:
    return ((1.0 - wC) / 2.0);

  case Shared_enum::Two_Pole_HP:
    return ((1.0 + wC) / 2.0);

  case Shared_enum::LP_Shelf:
    return (A * ((A + 1.0) - ((A - 1.0) * wC) - (beta * wS)));

  case Shared_enum::HP_Shelf:
    return (A * ((A + 1.0) + ((A - 1.0) * wC) - (beta * wS)));

  case Shared_enum::Bandpass:
    return (-alpha);

  case Shared_enum::Notch:
    return 1.0;

  case Shared_enum::Peaking:
    return (1.0 - (alpha * A));

  case Shared_enum::All_Pass:
    return (1.0 + alpha);

  case Shared_enum::None:
    return 0.0;

  default:
    return 0.0;
  }
}

float ADAU1466_Bilinear::coefa0(Shared_enum::tfilters filtertype)
{

  switch (filtertype)
  {
  case Shared_enum::One_Pole_LP:
    return 1.0;

  case Shared_enum::One_Pole_HP:
    return 1.0;

  case Shared_enum::Two_Pole_LP:
    return (1.0 + alpha);

  case Shared_enum::Two_Pole_HP:
    return (1.0 + alpha);

  case Shared_enum::LP_Shelf:
    return (((A + 1.0) + ((A - 1.0) * wC) + (beta * wS)));

  case Shared_enum::HP_Shelf:
    return (((A + 1.0) - ((A - 1.0) * wC) + (beta * wS)));

  case Shared_enum::Bandpass:
    return (1.0 + alpha);

  case Shared_enum::Notch:
    return (1.0 + alpha);

  case Shared_enum::Peaking:
    return (1.0 + (alpha / A));

  case Shared_enum::All_Pass:
    return (1.0 + alpha);

  case Shared_enum::None:
    return 1.0;

  default:
    return 1.0;
  }
}

float ADAU1466_Bilinear::coefa1(Shared_enum::tfilters filtertype)
{

  switch (filtertype)
  {
  case Shared_enum::One_Pole_LP:
    k = tan(w / 2.0);
    alpha = 1.0 + k;
    return (-((1.0 - k) / alpha));

  case Shared_enum::One_Pole_HP:
    k = tan(w / 2.0);
    alpha = 1.0 + k;
    return (-((1.0 - k) / alpha));

  case Shared_enum::Two_Pole_LP:
    return (-2.0 * wC);

  case Shared_enum::Two_Pole_HP:
    return (-2.0 * wC);

  case Shared_enum::LP_Shelf:
    return (-2.0 * ((A - 1.0) + ((A + 1.0) * wC)));

  case Shared_enum::HP_Shelf:
    return (2.0 * ((A - 1.0) - ((A + 1.0) * wC)));

  case Shared_enum::Bandpass:
    return (-2.0 * wC);

  case Shared_enum::Notch:
    return (-2.0 * wC);

  case Shared_enum::Peaking:
    return (-2.0 * wC);

  case Shared_enum::All_Pass:
    return (-2.0 * wC);

  case Shared_enum::None:
    return 0.0;

  default:
    return 0.0;
  }
}

float ADAU1466_Bilinear::coefa2(Shared_enum::tfilters filtertype)
{
  switch (filtertype)
  {
  case Shared_enum::One_Pole_LP:
    return 0.0;

  case Shared_enum::One_Pole_HP:
    return 0.0;

  case Shared_enum::Two_Pole_LP:
    return (1.0 - alpha);

  case Shared_enum::Two_Pole_HP:
    return (1.0 - alpha);

  case Shared_enum::LP_Shelf:
    return (((A + 1.0) + ((A - 1.0) * wC) - (beta * wS)));

  case Shared_enum::HP_Shelf:
    return (((A + 1.0) - ((A - 1.0) * wC) - (beta * wS)));

  case Shared_enum::Bandpass:
    return (1.0 - alpha);

  case Shared_enum::Notch:
    return (1.0 - alpha);

  case Shared_enum::Peaking:
    return (1.0 - (alpha / A));

  case Shared_enum::All_Pass:
    return (1.0 - alpha);

  case Shared_enum::None:
    return 0.0;

  default:
    return 0.0;
  }
}
